/* $Id: tcpipver.c 1586 2004-11-01 02:26:32Z bird $ */
/** @file
 *
 * LIBC SYS Backend - TCP/IP Version Specific Code.
 *
 * This file is included from tcpipver43.c with TCPV40HDRS defined,
 * this might be a little bit confusing but it saves a lot of coding.
 *
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#define INCL_TCPIP_ALLIOCTLS
#include "libc-alias.h"
#include <errno.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/tcp_var.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>
#include <net/route.h>

#include <sys/fcntl.h>
#include <sys/filio.h>
#include <sys/time.h>
#include <386/builtin.h>
#include <sys/smutex.h>
#include <emx/io.h>
#include <emx/startup.h>
#define INCL_BASE
#define INCL_FSMACROS
#include <os2emx.h>
#include <InnoTekLIBC/sharedpm.h>
#include <InnoTekLIBC/tcpip.h>
#include <InnoTekLIBC/fork.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_SOCKET
#include <InnoTekLIBC/logstrict.h>


/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#ifdef TCPV40HDRS

#define MY_FD_SET(fd, set)      FD_SET(fd, set)
#define MY_FD_ISSET(fd, set)    FD_ISSET(fd, set)
#define MY_FD_ZERO(set, cb)     bzero(set, cb);
#define my_fd_set               fd_set

#else

#define MY_FD_SET(fd, set)      V5_FD_SET(fd, set)
#define	MY_FD_ISSET(fd, set)    V5_FD_ISSET(fd, set)
#define MY_FD_ZERO(set, cb)     bzero(set, cb);
#define my_fd_set               v5_fd_set

#endif


/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
#ifdef TCPV40HDRS
/* not included in the TCPV40HDRS - this is from the BSD4.4 stack headers
                                    and doesn't have to right!!! */
struct arpreq_tr {
        struct  sockaddr arp_pa;                /* protocol address */
        struct  sockaddr arp_ha;                /* hardware address */
        long_int arp_flags;                     /* flags */
        u_short arp_rcf;                        /* token ring routing control field */
        u_short arp_rseg[8];                    /* token ring routing segments */
};

#define MAX_IN_MULTI      16*IP_MAX_MEMBERSHIPS      /* 320 max per os2 */
struct  addrreq  {                              /* get multicast addresses */
        char    ifr_name[IFNAMSIZ];
        struct  sockaddr ifr_addrs;
        u_long  maddr[MAX_IN_MULTI];
};

#pragma pack(1)
struct  oarptab {
        struct  in_addr at_iaddr;       /* internet address */
        u_char  at_enaddr[6];           /* ethernet address */
        u_char  at_timer;               /* minutes since last reference */
        u_char  at_flags;               /* flags */
#ifdef KERNEL
        struct  mbuf *at_hold;          /* last packet until resolved/timeout */
#else
        void * at_hold;
#endif
        u_short at_rcf;                 /* token ring routing control field */
        u_short at_rseg[8];             /* token ring routing segments */
        u_long  at_millisec;            /* TOD milliseconds of last update */
        u_short at_interface;           /* interface index */
};
#pragma pack()

#pragma pack(1)
struct  statatreq {
        u_long addr;
        short interface;
        u_long mask;
        u_long broadcast;
};
#pragma pack()

#endif

/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int TCPNAME(ops_Close)(PLIBCFH pFH, int fh);
static int TCPNAME(ops_Read)(PLIBCFH pFH, int fh, void *pvBuf, size_t cbRead, size_t *pcbRead);
static int TCPNAME(ops_Write)(PLIBCFH pFH, int fh, const void *pvBuf, size_t cbWrite, size_t *pcbWritten);
static int TCPNAME(ops_Duplicate)(PLIBCFH pFH, int fh, int *pfhNew);
static int TCPNAME(ops_FileControl)(PLIBCFH pFH, int fh, int iIOControl, int iArg, int *prc);
static int TCPNAME(ops_IOControl)(PLIBCFH pFH, int fh, int iIOControl, int iArg, int *prc);
static int TCPNAME(ops_Select)(int cFHs, struct fd_set *pRead, struct fd_set *pWrite, struct fd_set *pExcept, struct timeval *tv, int *prc);
static int TCPNAME(ops_ForkChild)(struct __libc_FileHandle *pFH, int fh, __LIBC_PFORKHANDLE pForkHandle, __LIBC_FORKOP enmOperation);

static int TCPNAME(imp_soclose)(int s);
#ifndef TCPV40HDRS
static int TCPNAME(imp_ioctl)(int, int, char *);
#endif
static int TCPNAME(imp_os2_ioctl)(int, unsigned long, char *, int);
static int TCPNAME(imp_recv)(int, void *, int, int);
static int TCPNAME(imp_send)(int, const void *, int, int);
static int TCPNAME(imp_removesocketfromlist)(int);
static int TCPNAME(imp_addsockettolist)(int);
#ifdef TCPV40HDRS
static int TCPNAME(imp_bsdselect)(int, fd_set *, fd_set *, fd_set *, struct timeval *);
#else
static int TCPNAME(imp_bsdselect)(int, v5_fd_set *, v5_fd_set *, v5_fd_set *, struct timeval *);
#endif

static void TCPNAME(Cleanup)(void);
static int  tcpipForkChild1(__LIBC_PFORKHANDLE pForkHandle, __LIBC_FORKOP enmOperation);


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** Head of sockets. */
static volatile PLIBCSOCKETFH   gpSocketsHead;
/** Spin mutex protecting the gpSocketsHead list. */
static _smutex                  gsmtxSockets;

/** Operations on sockets. */
static const __LIBC_FHOPS       gSocketOps =
{
#ifdef TCPV40HDRS
    enmFH_Socket43, /* Handle type. */
#else
    enmFH_Socket44, /* Handle type. */
#endif
    TCPNAME(ops_Close),
    TCPNAME(ops_Read),
    TCPNAME(ops_Write),
    TCPNAME(ops_Duplicate),
    TCPNAME(ops_FileControl),
    TCPNAME(ops_IOControl),
    TCPNAME(ops_Select),
    NULL /* fork parent */,
    TCPNAME(ops_ForkChild)
};


/** Close operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 */
static int TCPNAME(ops_Close)(PLIBCFH pFH, int fh)
{
    LIBCLOG_ENTER("pFH=%p:{iSocket=%d} fh=%d\n", (void *)pFH, ((PLIBCSOCKETFH)pFH)->iSocket, fh);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int             rc;

    /*
     * Dereference it and see if closing is required.
     */
    rc = __libc_spmSocketDeref(pSocketFH->iSocket);
    if (!rc)
    {
        rc = TCPNAME(imp_soclose)(pSocketFH->iSocket);
        if (rc)
        {
            rc = -__libc_TcpipGetSocketErrno();
            if (rc != -EBADF && rc != -ENOTSOCK && rc != -EALREADY)
            {
                LIBC_ASSERTM_FAILED("undocument errno from soclose() rc=%d\n", rc);
                /* Add reference again. the socket is probably not closed. */
                __libc_spmSocketRef(pSocketFH->iSocket);
            }
        }
    }
    else if (rc > 0)
    {
        /*
         * Someone is using the socket, we'll remove it from the list of
         * sockets owned by this process to prevent it from being closed.
         */
        rc = TCPNAME(imp_removesocketfromlist)(pSocketFH->iSocket);
        if (rc)
        {
            LIBC_ASSERTM(!rc, "removesocketfromlist(%d) -> rc=%d sock_errno()->%d\n", pSocketFH->iSocket, rc, TCPNAME(imp_sock_errno)());
            rc = -__libc_TcpipGetSocketErrno();
        }
    }
    else
        rc = -EBADF;

    /*
     * Unlink from our list.
     */
    if (rc != -EBADF && rc != -ENOTSOCK && rc != -EALREADY)
    {
        _smutex_request(&gsmtxSockets);
        if (pSocketFH->pNext)
            pSocketFH->pNext->pPrev = pSocketFH->pPrev;
        if (pSocketFH->pPrev)
            pSocketFH->pPrev->pNext = pSocketFH->pNext;
        else
            gpSocketsHead = pSocketFH->pNext;
        _smutex_release(&gsmtxSockets);
    }

    LIBCLOG_RETURN_INT(rc);
}


/** Read operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pvBuf       Pointer to the buffer to read into.
 * @param   cbRead      Number of bytes to read.
 * @param   pcbRead     Where to store the count of bytes actually read.
 */
static int TCPNAME(ops_Read)(PLIBCFH pFH, int fh, void *pvBuf, size_t cbRead, size_t *pcbRead)
{
    LIBCLOG_ENTER("pFH=%p:{iSocket=%d} fh=%d pvBuf=%p cbRead=%d pcbRead=%p\n",
                  (void *)pFH, ((PLIBCSOCKETFH)pFH)->iSocket, fh, pvBuf, cbRead, (void *)pcbRead);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int             rc;

    rc = TCPNAME(imp_recv)(pSocketFH->iSocket, pvBuf, cbRead, 0);
    if (rc < 0)
    {
        *pcbRead = 0;
        rc = -__libc_TcpipGetSocketErrno();
        LIBCLOG_RETURN_INT(rc);
    }

    *pcbRead = rc;
    LIBCLOG_RETURN_INT(0);
}

/** Write operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pvBuf       Pointer to the buffer which contains the data to write.
 * @param   cbWrite     Number of bytes to write.
 * @param   pcbWritten  Where to store the count of bytes actually written.
 */
static int TCPNAME(ops_Write)(PLIBCFH pFH, int fh, const void *pvBuf, size_t cbWrite, size_t *pcbWritten)
{
    LIBCLOG_ENTER("pFH=%p:{iSocket=%d} fh=%d pvBuf=%p cbWrite=%d pcbWritten=%p\n",
                  (void *)pFH, ((PLIBCSOCKETFH)pFH)->iSocket, fh, pvBuf, cbWrite, (void *)pcbWritten);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int             rc;

    rc = TCPNAME(imp_send)(pSocketFH->iSocket, pvBuf, cbWrite, 0);
    if (rc < 0)
    {
        *pcbWritten = 0;
        rc = -__libc_TcpipGetSocketErrno();
        LIBCLOG_RETURN_INT(rc);
    }

    *pcbWritten = rc;
    LIBCLOG_RETURN_INT(0);
}


/** Duplicate handle operation.
 * @returns 0 on success, OS/2 error code on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pfhNew      Where to store the duplicate filehandle.
 *                      The input value describe how the handle is to be
 *                      duplicated. If it's -1 a new handle is allocated.
 *                      Any other value will result in that value to be
 *                      used as handle. Any existing handle with that
 *                      value will be closed.
 */
static int TCPNAME(ops_Duplicate)(PLIBCFH pFH, int fh, int *pfhNew)
{
    LIBCLOG_ENTER("pFH=%p:{iSocket=%d} fh=%d pfhNew=%p\n", (void *)pFH, ((PLIBCSOCKETFH)pFH)->iSocket, fh, (void *)pfhNew);
    LIBC_ASSERTM_FAILED("Duplication of sockets aren't supported yet.\n");
    LIBCLOG_RETURN_INT(-ENOSYS);
}


/** File Control operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   iRequest    Which file file descriptior request to perform.
 * @param   iArg        Argument which content is specific to each
 *                      iRequest operation.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int TCPNAME(ops_FileControl)(PLIBCFH pFH, int fh, int iRequest, int iArg, int *prc)
{
    LIBCLOG_ENTER("pFH=%p:{iSocket=%d} fh=%d iRequest=%#x iArg=%#x prc=%p\n",
                  (void *)pFH, ((PLIBCSOCKETFH)pFH)->iSocket, fh, iRequest, iArg, (void *)prc);
    //PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int             rc = 0;


    /*
     * Service the request (very similar to __fcntl()!).
     */
    switch (iRequest)
    {
        /*
         * Get file status flags and access modes.
         */
        case F_GETFL:
        {
            unsigned fFlags = pFH->fFlags & (O_ACCMODE | O_APPEND | O_NONBLOCK | O_SYNC /*| O_*SYNC*/);
            *prc = fFlags;
            break;
        }

        /*
         * Set file status flags.
         */
        case F_SETFL:
        {
            /** @todo implement this properly. See FCNTLFLAGS. */
            LIBC_ASSERTM_FAILED("F_SETFL isn't implemented but returns success. arg=%#x\n", iArg);
            *prc = 0;
            break;
        }

        /*
         * Get file descriptor flags.
         */
        case F_GETFD:
            *prc = pFH->fFlags >> __LIBC_FH_FDFLAGS_SHIFT;
            break;

        /*
         * Set file descriptor flags.
         */
        case F_SETFD:
            if (iArg & ~(FD_CLOEXEC))
            {
                LIBC_ASSERTM_FAILED("Invalid argument %#x to F_SETFD\n", iArg);
                rc = -EINVAL;
                *prc = -1;
            }
            else
            {
                unsigned fFlags = pFH->fFlags;
                fFlags &= ~__LIBC_FH_FDFLAGS_MASK;
                fFlags |= iArg << __LIBC_FH_FDFLAGS_SHIFT;
                __atomic_xchg(&pFH->fFlags, fFlags);
                *prc = 0;
            }
            break;

        default:
            LIBC_ASSERTM_FAILED("Invalid or Unsupported request %#x\n", iRequest);
            rc = -EINVAL;
            *prc = -1;
            break;
    }

    LIBCLOG_RETURN_INT(rc);
}


/** I/O Control operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   iIOControl  Which I/O control operation to perform.
 * @param   iArg        Argument which content is specific to each
 *                      iIOControl operation.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int TCPNAME(ops_IOControl)(PLIBCFH pFH, int fh, int iIOControl, int iArg, int *prc)
{
    LIBCLOG_ENTER("pFH=%p:{iSocket=%d} fh=%d iIOControl=%#x iArg=%#x prc=%p\n",
                  (void *)pFH, ((PLIBCSOCKETFH)pFH)->iSocket, fh, iIOControl, iArg, (void *)prc);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    char           *pchArg = (char *)iArg;
    int             rc;
#ifndef TCPV40HDRS
    switch (__IOCLW(iIOControl))
    {
        case SIOSTATARP:
            rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct oarptab));
            break;
        case SIOSTATAT:
            /** this isn't really suitable for this ioctl interface!! */
            rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct statatreq) + 2);
            break;
        case SIOSTATIF:
            rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct ifmib));
            break;
        case SIOSTATIF42:
            /* What the h*** is the difference between the SIOSTATIF42 ioctl and SIOSTATIF?
               The docs doesn't make sense when looking in the headers... */
            rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct ifmib));
            break;
        case SIOSTATRT:
            rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct rtentries));
            break;
        case SIOSTATSO:
            /** this isn't really suitable for this ioctl interface!! */
            rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct sockaddr));
            break;

        case FIONBIO:
            rc = TCPNAME(imp_ioctl)(pSocketFH->iSocket, iIOControl, pchArg);
            if (rc)
            {
                if (*(unsigned*)pchArg)
                    pSocketFH->core.fFlags |= O_NDELAY;
                else
                    pSocketFH->core.fFlags &= ~O_NDELAY;
            }
            break;

        default:
            rc = TCPNAME(imp_ioctl)(pSocketFH->iSocket, iIOControl, pchArg);
            break;
    }

#else
    int     cb;

    /*
     * This ioctl interface requires a size, which thus is something we
     * must determin. :-/
     */
    switch (__IOCLW(iIOControl))
    {
        case __IOCLW(FIONREAD):
        case __IOCLW(FIOASYNC):
        case __IOCLW(SIOCATMARK):
        case __IOCLW(FIONBIO):
        case __IOCLW(SIOCSHIWAT):
        case __IOCLW(SIOCGHIWAT):
        case __IOCLW(SIOCSLOWAT):
        case __IOCLW(SIOCGLOWAT):
        case __IOCLW(SIOCSPGRP):
        case __IOCLW(SIOCGPGRP):
            cb = sizeof(int);
            break;

        case __IOCLW(SIOCSHOSTID):
        case __IOCLW(SIOCGNBNAME):
        case __IOCLW(SIOCSNBNAME):
        case __IOCLW(SIOCGNCBFN):
            cb = sizeof(long);
            break;

        case __IOCLW(FIONSTATUS):
            cb = sizeof(short) * 4;
            break;

        case __IOCLW(FIOBSTATUS):
            cb = sizeof(short);
            break;

        case __IOCLW(SIOCGIFADDR):
        case __IOCLW(SIOCGIFBRDADDR):
        case __IOCLW(SIOCGIFDSTADDR):
        case __IOCLW(SIOCGIFFLAGS):
        case __IOCLW(SIOCGIFMETRIC):
        case __IOCLW(SIOCGIFNETMASK):
        case __IOCLW(SIOCSIFADDR):
        case __IOCLW(SIOCSIFBRDADDR):
        case __IOCLW(SIOCSIFDSTADDR):
        case __IOCLW(SIOCSIFFLAGS):
        case __IOCLW(SIOCSIFMETRIC):
        case __IOCLW(SIOCSIFNETMASK):
        case __IOCLW(SIOCSIFALLRTB):
        case __IOCLW(SIOCSIF802_3):
        case __IOCLW(SIOCSIFNO802_3):
        case __IOCLW(SIOCSIFNOREDIR):
        case __IOCLW(SIOCSIFYESREDIR):
        case __IOCLW(SIOCSIFMTU):
        case __IOCLW(SIOCSIFFDDI):
        case __IOCLW(SIOCSIFNOFDDI):
        case __IOCLW(SIOCSRDBRD):
        case __IOCLW(SIOCADDMULTI):
        case __IOCLW(SIOCDELMULTI):
        case __IOCLW(SIOCMULTISBC):
        case __IOCLW(SIOCMULTISFA):
        case __IOCLW(SIOCDIFADDR):
        case __IOCLW(SIOCGUNIT):
        case __IOCLW(SIOCGIFVALID):
        case __IOCLW(SIOCGIFEFLAGS):
        case __IOCLW(SIOCSIFEFLAGS):
        case __IOCLW(SIOCGIFTRACE):
        case __IOCLW(SIOCSIFTRACE):
        case __IOCLW(SIOCSSTAT):
        case __IOCLW(SIOCGSTAT):
            cb = sizeof(struct ifreq);
            break;

        case __IOCLW(SIOCADDRT):
        case __IOCLW(SIOCDELRT):
            cb = sizeof(struct rtentry);
            break;

        case __IOCLW(SIOCSARP):
        case __IOCLW(SIOCGARP):
        case __IOCLW(SIOCDARP):
            cb = sizeof(struct arpreq);
            break;

        case __IOCLW(SIOCGIFCONF):
            cb = sizeof(struct ifconf);
            break;

        case __IOCLW(SIOCSARP_TR):
        case __IOCLW(SIOCGARP_TR):
            cb = sizeof(struct arpreq_tr);
            break;

        /* duplicate! Bug in BSD 4.3 stack I'd say.
        case __IOCLW(SIOCAIFADDR):
            cb = sizeof(struct ifaliasreq);
            break;
         */

        case __IOCLW(SIOCGIFBOUND):
            cb = sizeof(struct bndreq);
            break;

        case __IOCLW(SIOCGMCAST):
            cb = sizeof(struct addrreq);
            break;

        case __IOCLW(SIOSTATMBUF):
            cb = sizeof(struct mbstat);
            break;

        case __IOCLW(SIOSTATTCP):
            cb = sizeof(struct tcpstat);
            break;

        case __IOCLW(SIOSTATUDP):
            cb = sizeof(struct udpstat);
            break;

        case __IOCLW(SIOSTATIP):
            cb = sizeof(struct ipstat);
            break;

        case __IOCLW(SIOSTATSO):
            cb = sizeof(struct sockaddr);
            break;

        case __IOCLW(SIOSTATRT):
            cb = sizeof(struct rtentries);
            break;

        case __IOCLW(SIOFLUSHRT):
            cb = sizeof(long);
            break;

        case __IOCLW(SIOSTATICMP):
            cb = sizeof(struct icmpstat);
            break;

        case __IOCLW(SIOSTATIF):
            cb = sizeof(struct ifmib);
            break;

        case __IOCLW(SIOSTATAT):
            /** this isn't really suitable for this ioctl interface!! */
            cb = sizeof(struct statatreq) + 2;
            break;

        case __IOCLW(SIOSTATARP):
            cb = sizeof(struct oarptab);
            break;

        case __IOCLW(SIOSTATIF42):
            cb = sizeof(struct ifmib); /* docs are unclear here */
            break;

        case __IOCLW(SIOSTATCNTRT):
            cb = sizeof(int);
            break;

        case __IOCLW(SIOSTATCNTAT):
            cb = sizeof(int);
            break;

        default:
            cb = -1;
            break;

/** @todo There is a bunch of IOCtls in the BSD 4.3 stack
 *        for which I don't have the docs and don't know exact
 *        what sizes to use:
 *               FIOTCPCKSUM     ioc('f', 128)
 *               FIONURG         ioc('f', 121)
 *               SIOMETRIC1RT    ioc('r', 12)
 *               SIOMETRIC2RT    ioc('r', 13)
 *               SIOMETRIC3RT    ioc('r', 14)
 *               SIOMETRIC4RT    ioc('r', 15)
 *               SIOCREGADDNET   ioc('r', 12)
 *               SIOCREGDELNET   ioc('r', 13)
 *               SIOCREGROUTES   ioc('r', 14)
 *               SIOCFLUSHROUTES ioc('r', 15)
 *               SIOCSIFSETSIG   ioc('i', 25)
 *               SIOCSIFCLRSIG   ioc('i', 26)
 *               SIOCSIFBRD      ioc('i', 27)
 *               SIOCGIFLOAD     ioc('i', 27)
 *               SIOCSIFFILTERSRC ioc('i', 28)
 *               SIOCGIFFILTERSRC ioc('i',29)
 *               SIOCSIFSNMPSIG  ioc('i', 33)
 *               SIOCSIFSNMPCLR  ioc('i', 34)
 *               SIOCSIFSNMPCRC  ioc('i', 35)
 *               SIOCSIFPRIORITY ioc('i', 36)
 *               SIOCGIFPRIORITY ioc('i', 37)
 *               SIOCSIFFILTERDST ioc('i', 38)
 *               SIOCGIFFILTERDST ioc('i',39)
 *               SIOCSIFSPIPE     ioc('i',71)
 *               SIOCSIFRPIPE     ioc('i',72)
 *               SIOCSIFTCPSEG   ioc('i',73)
 *               SIOCSIFUSE576   ioc('i',74)
 */
    } /* size switch */

    /*
     * Do the call.
     */
    if (cb >= 0)
    {
        switch (__IOCLW(iIOControl))
        {
            case FIONBIO:
                rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, cb);
                if (rc)
                {
                    if (*(unsigned*)pchArg)
                        pSocketFH->core.fFlags |= O_NDELAY;
                    else
                        pSocketFH->core.fFlags &= ~O_NDELAY;
                }
                break;

            default:
                rc = TCPNAME(imp_os2_ioctl)(pSocketFH->iSocket, iIOControl, pchArg, cb);
                break;
        }
    }
    else
    {   /*  */
        *prc = -1;
        LIBCLOG_RETURN_INT(-ENOSYS);
    }

#endif

    *prc = rc;
    if (rc < 0)
    {
        rc = -__libc_TcpipGetSocketErrno();
        LIBCLOG_RETURN_INT(rc);
    }
    LIBCLOG_RETURN_INT(0);
}


/** Select operation.
 * The select operation is only performed if all handles have the same
 * select routine (the main worker checks this).
 *
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   cFHs        Range of handles to be tested.
 * @param   pRead       Bitmap for file handles to wait upon to become ready for reading.
 * @param   pWrite      Bitmap for file handles to wait upon to become ready for writing.
 * @param   pExcept     Bitmap of file handles to wait on (error) exceptions from.
 * @param   tv          Timeout value.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int TCPNAME(ops_Select)(int cFHs, struct fd_set *pRead, struct fd_set *pWrite, struct fd_set *pExcept, struct timeval *tv, int *prc)
{
    LIBCLOG_ENTER("cFHs=%d pRead=%p pWrite=%p pExcept=%p tv=%p prc=%p\n",
                  cFHs, (void *)pRead, (void *)pWrite, (void *)pExcept, (void *)tv, (void *)prc);
    int rc;
    int saved_errno = errno;
    rc = TCPNAMEG(bsdselect)(cFHs, pRead, pWrite, pExcept, tv);
    *prc = rc;
    rc = rc >= 0 ? 0 : -errno;
    errno = saved_errno;
    LIBCLOG_RETURN_INT(rc);
}


/** Fork notification - child context.
 * Only the __LIBC_FORK_OP_FORK_CHILD operation is forwarded atm.
 * If NULL it's assumed that no notifiction is needed.
 *
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH             Pointer to the handle structure to operate on.
 * @param   fh              It's associated filehandle.
 * @param   pForkHandle     The fork handle.
 * @param   enmOperation    The fork operation.
 */
static int TCPNAME(ops_ForkChild)(struct __libc_FileHandle *pFH, int fh, __LIBC_PFORKHANDLE pForkHandle, __LIBC_FORKOP enmOperation)
{
    LIBCLOG_ENTER("pFH=%p:{iSocket=%d} fh=%d\n", (void *)pFH, ((PLIBCSOCKETFH)pFH)->iSocket, fh);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int rc = __libc_spmSocketRef(pSocketFH->iSocket);
    LIBC_ASSERTM(rc > 0, "__libc_spmSocketDeref(%d) -> rc=%d\n", pSocketFH->iSocket, rc);
    LIBCLOG_RETURN_INT(0);
    rc = rc;
}





/**
 * Allocates file handle for a socket.
 *
 * @returns Pointer to socket handle on success.
 * @returns NULL and errno on failure.
 * @param   iSocket     The socket to allocate handle for.
 * @param   pfh         Where to store the filename.
 */
PLIBCSOCKETFH TCPNAMEG(AllocFH)(int iSocket, int *pfh)
{
    LIBCLOG_ENTER("iSocket=%d\n", iSocket);
    PLIBCSOCKETFH pFH = TCPNAMEG(AllocFHEx)(-1, iSocket, O_RDWR | F_SOCKET, 1, pfh);
    LIBCLOG_RETURN_P(pFH);
}


/**
 * Allocates file handle for a socket - extended version.
 *
 * The socket is not removed from the socket list when calling this
 * allocation function. This is because it's used by the inherit
 * processor where such actions are unnecessary.
 *
 * @returns Pointer to socket handle on success.
 * @returns NULL and errno on failure.
 * @param   fh          The requested file handle.
 *                      If -1 any file handle can be used.
 * @param   iSocket     The socket to allocate handle for.
 * @param   fFlags      LIBC file handle flags.
 * @param   fNew        If set we handle the socket as brand new setting the global
 *                      reference counter to 1 and remove it from the TCPIP cleanup
 *                      list for the current process.
 *                      If clear we only increment the global reference counter.
 * @param   pfh         Where to store the filename.
 */
PLIBCSOCKETFH TCPNAMEG(AllocFHEx)(int fh, int iSocket, unsigned fFlags, int fNew, int *pfh)
{
    LIBCLOG_ENTER("iSocket=%d\n", iSocket);
    PLIBCFH         pFHLibc;
    int rc = __libc_FHAllocate(fh, fFlags, sizeof(LIBCSOCKETFH), &gSocketOps, &pFHLibc, pfh);
    if (!rc)
    {
        /*
         * Init the handle.
         */
        PLIBCSOCKETFH   pFH = (PLIBCSOCKETFH)pFHLibc;
        pFH->iSocket = iSocket;

        /*
         * Insert into the list.
         */
        pFH->pPrev = NULL;
        _smutex_request(&gsmtxSockets);
        pFH->pNext = gpSocketsHead;
        if (pFH->pNext)
            gpSocketsHead->pPrev = pFH;
        gpSocketsHead = pFH;
        _smutex_release(&gsmtxSockets);

        /*
         * New: Set usage counter to 1.
         * Existing: Increment reference counter and add to socket list.
         */
        if (fNew)
        {
            rc = __libc_spmSocketNew(iSocket);
            LIBC_ASSERTM(!rc, "__libc_spmSocketNew(%d) -> rc=%d\n", iSocket, rc);
        }
        else
        {
            rc = __libc_spmSocketRef(iSocket);
            LIBC_ASSERTM(rc > 0, "__libc_spmSocketRef(%d) -> rc=%d\n", iSocket, rc);
            rc = TCPNAME(imp_addsockettolist)(iSocket);
            LIBC_ASSERTM(!rc, "addsockettolist(%d) -> rc=%d\n", iSocket, rc);
        }
        LIBCLOG_RETURN_P(pFH);
    }
    else
        __libc_TcpipSetSocketErrno();

    LIBCLOG_RETURN_P(NULL);
}



/**
 * Calculate the size required for the converted set structure.
 * @returns number of bytes.
 * @param   c       Number of file descriptors specified in the call.
 * @param   pFrom   The select input fd_set structure.
 * @parm    pcFDs   Store the new number of file descriptors (socket handles) to examin.
 */
static inline int calcsize(int c, const struct fd_set *pFrom, int *pcFDs)
{
    int cbRet;
    int i;
#ifdef TCPV40HDRS
    /* here we need to figure out the max real socket handle */
    int iMax;
    for (iMax = *pcFDs - 1, i = 0; i < c; i++)
        if (FD_ISSET(i, pFrom))
        {
            PLIBCSOCKETFH   pFHSocket = __libc_TcpipFH(i);
            if (pFHSocket && iMax < pFHSocket->iSocket)
                iMax = pFHSocket->iSocket;
        }
    cbRet = (iMax + 8) / 8;             /* iMax is index not count, thus +8 not +7. */
    *pcFDs = iMax + 1;
#else
    for (cbRet = sizeof(u_short), i = 0; i < c; i++)
        if (FD_ISSET(i, pFrom))
            cbRet++;
    *pcFDs = c;
#endif
    if (cbRet < sizeof(struct my_fd_set))
        return sizeof(struct my_fd_set);
    return cbRet;
}

/** Converts the LIBC fd_set strcture pointed to by pFrom with it's LIBC socket handles,
 * to the fd_set strcuture used by the OS/2 tcpip and the OS/2 socket handles.
 * @returns 0 on success.
 * @returns -1 on failure, both errnos set.
 * @param   c       Number of file descriptors specified in the call.
 * @param   cb      Size of pTo. (used for zeroing it)
 * @param   pFrom   The select input fd_set structure.
 * @param   pTo     The structure we present to OS/2 TCPIP.
 *                  This will be initialized.
 * @param   pszType Typestring to use in the log.
 */
static inline int convert(int c, int cb, const struct fd_set *pFrom, struct my_fd_set *pTo, const char *pszType)
{
    int i;
    MY_FD_ZERO(pTo, cb)
    for (i = 0; i < c; i++)
    {
        if (FD_ISSET(i, pFrom))
        {
            PLIBCSOCKETFH   pFHSocket = __libc_TcpipFH(i);
            if (!pFHSocket)
            {
                LIBCLOG_MSG2("Invalid handle %d specified (%s).\n", i, pszType);
                return -1;
            }
            MY_FD_SET(pFHSocket->iSocket, pTo);
            LIBCLOG_MSG2("%s: %02d -> %03d\n", pszType, i, pFHSocket->iSocket);
        }
    }
    pszType = pszType;
    return 0;
}

/**
 * Updates the pTo structure with the fds marked ready in pFrom.
 *
 * @param   c       Number of file descriptors specified in the call.
 * @param   pFrom   The structure returned from OS/2 TCPIP select.
 * @param   pTo     The structure passed in to select which have to
 *                  be updated for the return.
 * @param   pszType Typestring to use in the log.
 */
static inline void update(int c, const struct my_fd_set *pFrom, struct fd_set *pTo, const char *pszType)
{
    int i;
    for (i = 0; i < c; i++)
    {
        if (FD_ISSET(i, pTo))
        {
            PLIBCSOCKETFH   pFHSocket = __libc_TcpipFH(i);
            if (pFHSocket)
            {
                if (!MY_FD_ISSET(pFHSocket->iSocket, pFrom))
                {
                    FD_CLR(i, pTo);
                    LIBCLOG_MSG2("%s: %02d -> %03d set\n", pszType, i, pFHSocket->iSocket);
                }
                else
                    LIBCLOG_MSG2("%s: %02d -> %03d clear\n", pszType, i, pFHSocket->iSocket);
            }
        }
    }
    pszType = pszType;
}



int TCPNAMEG(bsdselect)(int nfds, struct fd_set *readfds, struct fd_set *writefds, struct fd_set *exceptfds, struct timeval *tv)
{
    LIBCLOG_ENTER("nfds=%d readfds=%p writefds=%p exceptfds=%p tv=%p={tv_sec=%ld, tv_usec=%ld}\n",
                  nfds, (void *)readfds, (void *)writefds, (void *)exceptfds, (void *)tv, tv ? tv->tv_sec : -1, tv ? tv->tv_usec : -1);
#ifdef TCPV40HDRS
    struct fd_set      *pRead;
    struct fd_set      *pWrite;
    struct fd_set      *pExcept;
#else
    struct v5_fd_set   *pRead;
    struct v5_fd_set   *pWrite;
    struct v5_fd_set   *pExcept;
#endif
    int                 rc;
    int                 cb = 0;
    int                 cFDs = 0;

    /*
     * Calculate bitmapsize.
     */
    if (readfds)
        cb = calcsize(nfds, readfds, &cFDs);
    if (writefds)
    {
        int cbT = calcsize(nfds, writefds, &cFDs);
        if (cbT > cb)
            cb = cbT;
    }
    if (exceptfds)
    {
        int cbT = calcsize(nfds, exceptfds, &cFDs);
        if (cbT > cb)
            cb = cbT;
    }

    /*
     * Allocate new bitmaps.
     */
    pRead = NULL;
    if (readfds)
    {
        pRead = alloca(cb);
        if (!pRead)
        {
            __libc_TcpipSetErrno(ENOMEM);
            LIBCLOG_RETURN_INT(-1);
        }
    }

    pWrite = NULL;
    if (writefds)
    {
        pWrite = alloca(cb);
        if (!pWrite)
        {
            __libc_TcpipSetErrno(ENOMEM);
            LIBCLOG_RETURN_INT(-1);
        }
    }

    pExcept = NULL;
    if (exceptfds)
    {
        pExcept = alloca(cb);
        if (!pExcept)
        {
            __libc_TcpipSetErrno(ENOMEM);
            LIBCLOG_RETURN_INT(-1);
        }
    }

    /*
     * Convert the bitmaps.
     */
    if (readfds)
    {
        if (convert(nfds, cb, readfds, pRead, "read "))
            LIBCLOG_RETURN_INT(-1);
    }

    if (writefds)
    {
        if (convert(nfds, cb, writefds, pWrite, "write"))
            LIBCLOG_RETURN_INT(-1);
    }

    if (exceptfds)
    {
        if (convert(nfds, cb, exceptfds, pExcept, "excpt"))
            LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Do the call.
     */
    LIBCLOG_MSG("calling native: cFDs=%d pRead=%p pWrite=%p, pExcept=%p tv=%p\n",
                cFDs, (void *)pRead, (void *)pWrite, (void *)pExcept, (void *)tv);
    rc = TCPNAME(imp_bsdselect)(cFDs, pRead, pWrite, pExcept, tv);
    if (rc < 0)
    {
        __libc_TcpipUpdateErrno();
        LIBCLOG_RETURN_INT(rc);
    }

    /*
     * Timeout?
     *  Just clear the bitmaps and return.
     */
    if (rc == 0)
    {
        cb = (nfds + 7) / 8;
        if (readfds)
            bzero(readfds, cb);
        if (writefds)
            bzero(writefds, cb);
        if (exceptfds)
            bzero(exceptfds, cb);
        LIBCLOG_RETURN_INT(0);
    }

    /*
     * Convert the bitmaps.
     */
    if (readfds)
        update(nfds, pRead, readfds, "read ");
    if (writefds)
        update(nfds, pWrite, writefds, "write");
    if (exceptfds)
        update(nfds, pExcept, exceptfds, "excpt");

    LIBCLOG_RETURN_INT(rc);
}





//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//
//
//          DYNAMIC IMPORTS
//
//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//\\//
#ifdef TCPV40HDRS
# define ORD_SOCK_ERRNO                 20
# define ORD_SET_ERRNO                  35
# define ORD_SOCLOSE                    17
# define ORD_IOCTL                      8
# define ORD_OS2_IOCTL                  8
# define ORD_RECV                       10
# define ORD_SEND                       13
# define ORD_ADDSOCKETTOLIST            27
# define ORD_REMOVESOCKETFROMLIST       28
# define ORD_BSDSELECT                  32
#else
# define ORD_SOCK_ERRNO                 20
# define ORD_SET_ERRNO                  35
# define ORD_SOCLOSE                    17
# define ORD_IOCTL                      8
# define ORD_OS2_IOCTL                  200
# define ORD_RECV                       10
# define ORD_SEND                       13
# define ORD_ADDSOCKETTOLIST            27
# define ORD_REMOVESOCKETFROMLIST       28
# define ORD_BSDSELECT                  32
#endif

/** Handle of the socket dll. */
static HMODULE      ghmod;
/** The name of the DLL. */
#ifdef TCPV40HDRS
static const char   gszDllName[] = "SO32DLL";
#else
static const char   gszDllName[] = "TCPIP32";
#endif


/**
 * Resolves an external symbol in a tcpip dll.
 */
static int  TCPNAME(get_imp)(unsigned iOrdinal, void **ppfn)
{
    int     rc;
    PFN     pfn;
    if (!ghmod)
    {

        HMODULE hmod;
        char    szErr[16];
        rc = DosLoadModule((PSZ)szErr, sizeof(szErr), (PCSZ)gszDllName, &hmod);
        if (rc)
        {
            LIBC_ASSERTM_FAILED("DosLoadModule(,,%s,) failed. rc=%d szErr=%.16s\n", gszDllName, rc, szErr);
            errno = ENOSYS;
            return -1;
        }
        __atomic_xchg((unsigned *)(void *)&ghmod, (unsigned)hmod);
    }

    /*
     * Resolve the symbol.
     */
    rc = DosQueryProcAddr(ghmod, iOrdinal, NULL, &pfn);
    if (rc)
    {
        LIBC_ASSERTM_FAILED("DosQueryProcAddr(%#lx,%d,,) failed. rc=%d\n", ghmod, iOrdinal, rc);
        errno = ENOSYS;
        return -1;
    }

    __atomic_xchg((unsigned *)ppfn, (unsigned)*pfn);
    return 0;
}

int     TCPNAME(imp_sock_errno)(void)
{
    static int (*TCPCALL pfn)(void);
    if (!pfn && TCPNAME(get_imp)(ORD_SOCK_ERRNO, (void **)(void *)&pfn))
        return errno;
    return pfn();
}

void    TCPNAME(imp_set_errno)(int iErr)
{
    static int (*TCPCALL pfn)(int iErr);
    if (!pfn && TCPNAME(get_imp)(ORD_SET_ERRNO, (void **)(void *)&pfn))
        return;
    pfn(iErr);
}

static int     TCPNAME(imp_soclose)(int s)
{
    static int (*TCPCALL pfn)(int s);
    if (!pfn && TCPNAME(get_imp)(ORD_SOCLOSE, (void **)(void *)&pfn))
        return -1;
    return pfn(s);
}

#ifndef TCPV40HDRS
static int     TCPNAME(imp_ioctl)(int s, int req, char *arg)
{
    static int (*TCPCALL pfn)(int s, int req, char *arg);
    if (!pfn && TCPNAME(get_imp)(ORD_IOCTL, (void **)(void *)&pfn))
        return -1;
    return pfn(s, req, arg);
}
#endif

static int     TCPNAME(imp_os2_ioctl)(int s, unsigned long req, char *arg, int len)
{
    static int (*TCPCALL pfn)(int s, unsigned long req, char *arg, int len);
    if (!pfn && TCPNAME(get_imp)(ORD_OS2_IOCTL, (void **)(void *)&pfn))
        return -1;
    return pfn(s,req, arg, len);
}

static int     TCPNAME(imp_recv)(int s, void *buf, int len, int flags)
{
    static int (*TCPCALL pfn)(int s, void *buf, int len, int flags);
    if (!pfn && TCPNAME(get_imp)(ORD_RECV, (void *)(void **)(void *)&pfn))
        return -1;
    return pfn(s, buf, len, flags);
}

static int     TCPNAME(imp_send)(int s, const void *buf, int len, int flags)
{
    static int (*TCPCALL pfn)(int s, const void *buf, int len, int flags);
    if (!pfn && TCPNAME(get_imp)(ORD_SEND, (void **)(void *)&pfn))
        return -1;
    return pfn(s, buf, len, flags);
}

static int     TCPNAME(imp_addsockettolist)(int s)
{
    static int (*TCPCALL pfn)(int s);
    if (!pfn && TCPNAME(get_imp)(ORD_ADDSOCKETTOLIST, (void **)(void *)&pfn))
        return -1;
    return pfn(s);
}

static int     TCPNAME(imp_removesocketfromlist)(int s)
{
    static int (*TCPCALL pfn)(int s);
    if (!pfn && TCPNAME(get_imp)(ORD_REMOVESOCKETFROMLIST, (void **)(void *)&pfn))
        return -1;
    return pfn(s);
}

#ifdef TCPV40HDRS
static int     TCPNAME(imp_bsdselect)(int nfds, fd_set *r, fd_set *w, fd_set *x, struct timeval *tv)
{
    static int (*TCPCALL pfn)(int nfds, fd_set *r, fd_set *w, fd_set *x, struct timeval *tv);
#else
static int     TCPNAME(imp_bsdselect)(int nfds, v5_fd_set *r, v5_fd_set *w, v5_fd_set *x, struct timeval *tv)
{
    static int (*TCPCALL pfn)(int nfds, v5_fd_set *r, v5_fd_set *w, v5_fd_set *x, struct timeval *tv);
#endif
    if (!pfn && TCPNAME(get_imp)(ORD_REMOVESOCKETFROMLIST, (void **)(void *)&pfn))
        return -1;
    return pfn(nfds, r, w, x, tv);
}



#undef  __LIBC_LOG_GROUP
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_INITTERM
_CRT_INIT1(tcpipInit)

/**
 * Init function which registers the tcpip cleanup handler with the exit list.
 */
static void tcpipInit(void)
{
    LIBCLOG_ENTER("\n");
    static void *pfn = (void*)tcpipInit; /* reference hack */
    pfn = pfn;
    __libc_spmSocketRegTerm(TCPNAME(Cleanup));
    LIBCLOG_RETURN_VOID();
}


/**
 * This function will close all open sockets updating
 * the global reference counters.
 *
 * The SPM exit list handler calls it.
 */
static void TCPNAME(Cleanup)(void)
{
    LIBCLOG_ENTER("\n");
    PLIBCSOCKETFH   pFH;

    /*
     * Walk list of socket handles and dereference them.
     */
    pFH = gpSocketsHead;
    while (pFH)
    {
        int rc = __libc_spmSocketDeref(pFH->iSocket);
        if (!rc)
        {
            /*
             * Close the socket, noone needs it.
             */
            rc = TCPNAME(imp_soclose)(pFH->iSocket);
            LIBC_ASSERTM(!rc, "soclose(%d) -> rc=%d sock_errno()->%d\n", pFH->iSocket, rc, TCPNAME(imp_sock_errno)());
        }
        else
        {
            /*
             * If in use by other process, we'll remove it from the list of sockets
             * owned by this process to prevent it from begin closed.
             */
            rc = TCPNAME(imp_removesocketfromlist)(pFH->iSocket);
            LIBC_ASSERTM(!rc, "removesocketfromlist(%d) -> rc=%d sock_errno()->%d\n", pFH->iSocket, rc, TCPNAME(imp_sock_errno)());
        }

        /* next */
        pFH = pFH->pNext;
    }
    LIBCLOG_RETURN_VOID();
}


#undef  __LIBC_LOG_GROUP
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_FORK

_FORK_CHILD1(0xffffff00, tcpipForkChild1)

/**
 * Reload the DLL in the child process.
 *
 * @returns 0 on success.
 * @returns negative errno on failure.
 * @param   pForkHandle     Pointer to fork handle.
 * @param   enmOperation    Callback operation.
 */
static int tcpipForkChild1(__LIBC_PFORKHANDLE pForkHandle, __LIBC_FORKOP enmOperation)
{
    LIBCLOG_ENTER("pForkHandle=%p enmOperation=%d\n", (void *)pForkHandle, enmOperation);
    int rc;
    switch (enmOperation)
    {
        case __LIBC_FORK_OP_FORK_CHILD:
        {
            rc = 0;
            if (ghmod)
                rc = __libc_tcpipForkLoadModule(ghmod, gszDllName);
            break;
        }

        default:
            rc = 0;
            break;
    }

    LIBCLOG_RETURN_INT(rc);
}


