/* $Id: backend.h 1574 2004-10-10 11:07:40Z bird $ */
/** @file
 *
 * LIBC - Backend header.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __InnoTekLIBC_backend_h__
#define __InnoTekLIBC_backend_h__

#include <sys/cdefs.h>
#include <sys/types.h>
#include <emx/io.h>

__BEGIN_DECLS

#ifndef __LIBC_THREAD_DECLARED
#define __LIBC_THREAD_DECLARED
typedef struct __libc_thread *__LIBC_PTHREAD, **__LIBC_PPTHREAD;
#endif
struct statfs;
struct stat;


/** @defgroup __libc_Back_thread   LIBC Backend - Threads
 * @{ */

/**
 * Initiatlize a new thread structure.
 *
 * @param   pThrd       Pointer to the thread structure.
 * @param   pParentThrd Pointer to the thread structure for the parent thread.
 *                      If NULL and thread id is 1 then inherit from parent process.
 *                      If NULL and thread is not null or no record of parent then
 *                      use defaults.
 */
void __libc_Back_threadInit(__LIBC_PTHREAD pThrd, const __LIBC_PTHREAD pParentThrd);

/**
 * Called before the thread structure is freed so the backend
 * can cleanup its members.
 *
 * @param   pThrd   Pointer to the thread in  question.
 */
void __libc_Back_threadCleanup(__LIBC_PTHREAD pThrd);

/**
 * Called in the context of the newly started thread to register
 * exception handler and to do other init stuff.
 *
 * @param   pExpRegRec  Exception handler registration record on the stack.
 *                      To be used for any exception handler registration.
 */
void __libc_Back_threadStartup(void *pExpRegRec);

/**
 * Called in the context of the thread which is to be terminated to
 * unregister exception handler and to do other final term stuff.
 *
 * @param   pExpRegRec  Exception handler registration record on the stack.
 *                      To be used for any exception handler registration.
 * @remark  This is called after __libc_Back_threadCleanup().
 * @remark  It is not called by thread which calls _endthread(), nor for the
 *          main thread.
 */
void __libc_Back_threadEnd(void *pExpRegRec);

/** @} */


/** @defgroup __libc_Back_fs   LIBC Backend - File System
 * @{ */

/**
 * Get the statistics for the filesystem which pszPath is located on.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     The path to somewhere in the filesystem.
 * @param   pStatFs     Where to store the obtained information.
 */
int __libc_Back_fsStat(const char *pszPath, struct statfs *pStatFs);

/**
 * Get file system statistics
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   fh          The filehandle of any file within the mounted file system.
 * @param   pStatFs     Where to store the statistics.
 */
int __libc_Back_fsStatFH(int fh, struct statfs *pStatFs);

/**
 * Get the statistics for all the mounted filesystems.
 *
 * @returns Number of returned statfs structs on success.
 * @returns Number of mounted filesystems on success if paStatFS is NULL
 * @returns Negative error code (errno.h) on failure.
 * @param   paStatFs    Where to to store the statistics.
 * @param   cStatFS     Number of structures the array pointed to by paStatFs can hold.
 * @param   fFlags      Flags, currently ignored.
 */
int __libc_Back_fsStats(struct statfs *paStatFs, unsigned cStatFs, unsigned fFlags);

/**
 * Schedules all file system buffers for writing.
 *
 * See sync() for OS/2 limitations.
 */
void __libc_Back_fsSync(void);

/**
 * Resolves the path into an canonicalized absolute path.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     The path to resolve.
 * @param   pszBuf      Where to store the resolved path.
 * @param   cchBuf      Size of the buffer.
 * @param   fFlags      Combination of __LIBC_BACKFS_FLAGS_RESOLVE_* defines.
 */
int __libc_Back_fsPathResolve(const char *pszPath, char *pszBuf, size_t cchBuf, unsigned fFlags);
/** Flags for __libc_Back_fsPathResolve().
 * @{ */
#define __LIBC_BACKFS_FLAGS_RESOLVE_FULL    0
/** Get the native path instead, no unix root translations. */
#define __LIBC_BACKFS_FLAGS_RESOLVE_NATIVE  0x10
/** @} */


/**
 * Changes the default drive of the process.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   chDrive     New default drive.
 */
int __libc_Back_fsDriveDefaultSet(char chDrive);

/**
 * Gets the default drive of the process.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pchDrive    Where to store the default drive.
 */
int __libc_Back_fsDriveDefaultGet(char *pchDrive);

/**
 * Sets or change the unixroot of the current process.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszNewRoot  The new root.
 */
int __libc_Back_fsDirChangeRoot(const char *pszNewRoot);

/**
 * Gets the current directory of the process on a
 * specific drive or on the current one.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     Where to store the path to the current directory.
 *                      This will be prefixed with a drive letter if we're
 *                      not in the unix tree.
 * @param   cchPath     The size of the path buffer.
 * @param   chDrive     The drive letter of the drive to get it for.
 *                      If '\0' the current dir for the current drive is returned.
 * @param   fFlags      Flags for skipping drive letter and slash.
 */
int __libc_Back_fsDirCurrentGet(char *pszPath, size_t cchPath, char chDrive, int fFlags);

/** Flags for __libc_Back_fsDirCurrentGet().
 * @{ */
#define __LIBC_BACK_FSCWD_NO_DRIVE          1
#define __LIBC_BACK_FSCWD_NO_ROOT_SLASH     2
/** @} */

/**
 * Changes the current directory of the process.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     Path to the new current directory of the process.
 */
int __libc_Back_fsDirCurrentSet(const char *pszPath);

/**
 * Creates a new directory.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     Path of the new directory.
 * @param   Mode        Permissions on the created directory.
 */
int __libc_Back_fsDirCreate(const char *pszPath, mode_t Mode);

/**
 * Removes a new directory.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     Path to the directory which is to be removed.
 */
int __libc_Back_fsDirRemove(const char *pszPath);

/**
 * Creates a symbolic link.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszTarget   The target of the symlink link.
 * @param   pszSymlink  The path to the symbolic link to create.
 */
int __libc_Back_fsSymlinkCreate(const char *pszTarget, const char *pszSymlink);

/**
 * Reads the content of a symbolic link.
 *
 * This is weird interface as it will return a truncated result if not
 * enough buffer space. It is also weird in that there is no string
 * terminator.
 *
 * @returns Number of bytes returned in pachBuf.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     The path to the symlink directory.
 * @param   pachBuf     Where to store the symlink value.
 * @param   cchBuf      Size of buffer.
 */
int __libc_Back_fsSymlinkRead(const char *pszPath, char *pachBuf, size_t cchBuf);

/**
 * Stats a symbolic link.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     Path to the file to stat. If this is a symbolic link
 *                      the link it self will be stat'ed.
 * @param   pStat       Where to store the file stats.
 */
int __libc_Back_fsSymlinkStat(const char *pszPath, struct stat *pStat);

/**
 * Stats a file.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszPath     Path to the file to stat.
 * @param   pStat       Where to store the file stats.
 */
int __libc_Back_fsFileStat(const char *pszPath, struct stat *pStat);

/**
 * Gets the file stats for a file by filehandle.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure. The content
 *          of *pStat is undefined.
 * @param   fh      Handle to file.
 * @param   pStat   Where to store the stats.
 */
int __libc_Back_fsFileStatFH(int fh, struct stat *pStat);


/** @defgroup __libc_Back_io   LIBC Backend - I/O and File Management.
 * @{ */

/**
 * Opens a file.
 *
 * @returns Filehandle to the opened file on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszFile     Path to the file.
 * @param   fFlags      Open flags.
 * @param   cbInitial   Initial filesize.
 * @param   Mode        The specified permission mask.
 * @param   fLibc       LIBC filehandle flags.
 * @param   ppFH        Where to store the LIBC filehandle structure which was created
 *                      for the opened file.
 */
int __libc_Back_ioFileOpen(const char *pszFile, int fFlags, off_t cbInitial, mode_t Mode, unsigned fLibc, PLIBCFH *ppFH);

/**
 * Change the current position of a file stream and get the new position.
 *
 * @returns new file offset on success.
 * @returns Negative error code (errno) on failure.
 * @param   hFile       File handle to preform seek operation on.
 * @param   off         Offset to seek to.
 * @param   iMethod     The seek method. SEEK_CUR, SEEK_SET or SEEK_END.
 */
off_t __libc_Back_ioSeek(int hFile, off_t off, int iMethod);

/**
 * Sets the size of an open file.
 *
 * When expanding a file the contents of the allocated
 * space is undefined.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   fh      Handle to the file which size should be changed.
 * @param   cbFile  The new filesize.
 * @param   fZero   If set any new allocated file space will be
 *                  initialized to zero.
 */
int __libc_Back_ioFileSizeSet(int fh, __off_t cbFile, int fZero);

/**
 * Try resolve a filehandle to a path.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   fh          The file handle.
 * @param   pszPath     Where to store the path.
 * @param   cchPath     The size of he buffer pointed to by pszPath.
 */
int __libc_Back_ioFHToPath(int fh, char *pszPath, size_t cchPath);

/** @} */

/** @} */


/** @defgroup __libc_Back_ldr   LIBC Backend - Loader
 * @{ */

/**
 * Opens a shared library.
 *
 * @returns 0 on success.
 * @returns Native error number.
 * @param   pszLibrary      Name of library to load.
 * @param   fFlags          Flags - ignored.
 * @param   ppvModule       Where to store the handle.
 * @param   pszError        Where to store error information.
 * @param   cchError        Size of error buffer.
 */
int  __libc_Back_ldrOpen(const char *pszLibrary, int fFlags, void **ppvModule, char *pszError, size_t cchError);

/**
 * Finds a symbol in an open shared library.
 *
 * @returns 0 on success.
 * @returns Native error number.
 * @param   pvModule        Module handle returned by __libc_Back_ldrOpen();
 * @param   pszSymbol       Name of the symbol we're to find in pvModule.
 * @param   ppfn            Where to store the symbol address.
 */
int __libc_Back_ldrSymbol(void *pvHandle,  const char *pszSymbol, void **ppfn);

/**
 * Closes a shared library.
 *
 * @returns 0 on success.
 * @returns Native error number.
 * @param   pvModule        Module handle returned by __libc_Back_ldrOpen();
 */
int  __libc_Back_ldrClose(void *pvModule);


/** @} */





/** @defgroup __libc_Back_misc   LIBC Backend - Miscellaneous
 * @{ */

/**
 * Gets the system load averages.
 * The load is the average values of ready and running threads(/processes)
 * over the last 1, 5 and 15 minuttes.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pardAvgs    Where to store the samples.
 * @param   cAvgs       Number of samples to get. Max is 3.
 * @remark  See OS/2 limitations in getloadavg().
 */
int __libc_Back_miscLoadAvg(double *pardAvgs, unsigned cAvgs);

/** @} */


/** @defgroup __libc_Back_Signals   LIBC Backend - Signals and Exceptions
 * @{ */

#if defined(END_OF_CHAIN) && defined(INCL_DOSEXCEPTIONS)
/**
 * The LIBC Sys Backend exception handler.
 *
 * @returns XCPT_CONTINUE_SEARCH or XCPT_CONTINUE_EXECUTION.
 * @param   pXcptRepRec     Report record.
 * @param   pXcptRegRec     Registration record.
 * @param   pCtx            Context record.
 * @param   pvWhatEver      Not quite sure what this is...
 */
ULONG _System __libc_Back_exceptionHandler(PEXCEPTIONREPORTRECORD       pXcptRepRec,
                                           PEXCEPTIONREGISTRATIONRECORD pXcptRegRec,
                                           PCONTEXTRECORD               pCtx,
                                           PVOID                        pvWhatEver);
#endif

/** @} */

__END_DECLS

#endif
