/* $Id: fs.c 1534 2004-10-01 04:55:39Z bird $ */
/** @file
 *
 * LIBC SYS Backend - file system stuff.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird@innotek.de>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_FSMACROS
#define INCL_BASE
#define INCL_ERRORS
#include <os2emx.h>
#include "fs.h"
#define __USE_GNU
#include <stddef.h>
#include <string.h>
#include <errno.h>
#include <386/builtin.h>
#include <sys/stat.h>
#include <sys/fmutex.h>
#include <emx/startup.h>
#include "syscalls.h"
#include <InnoTekLIBC/sharedpm.h>
#include <InnoTekLIBC/pathrewrite.h>
#include <InnoTekLIBC/libc.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_FS
#include <InnoTekLIBC/logstrict.h>


/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#define ORD_DOS32OPENL              981
#define ORD_DOS32SETFILEPTRL        988
#define ORD_DOS32SETFILESIZEL       989
#define ORD_DOS32SETFILELOCKSL      986

#define SIZEOF_ACHBUFFER    (sizeof(FILEFINDBUF4) + 32)


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** Mutex protecting all the FS globals.
 * __libc_gfsUMask and the API pointers are not protected by this semaphore. */
static _fmutex __libc_gmtxFS;

/** Indicator whether or not we're in the unix tree. */
int     __libc_gfInUnixTree = 0;
/** Length of the unix root if the unix root is official. */
int     __libc_gcchUnixRoot = 0;
/** The current unix root. */
char    __libc_gszUnixRoot[PATH_MAX];
/** The umask of the process. */
mode_t  __libc_gfsUMask = 022;


ULONG (* _System __libc_gpfnDosOpenL)(PCSZ pszFileName, PHFILE phFile, PULONG pulAction, LONGLONG llFileSize, ULONG ulAttribute, ULONG ulOpenFlags, ULONG ulOpenMode, PEAOP2 pEABuf) = NULL;
ULONG (* _System __libc_gpfnDosSetFilePtrL)(HFILE hFile, LONGLONG llOffset, ULONG ulOrigin, PLONGLONG pllPos) = NULL;
ULONG (* _System __libc_gpfnDosSetFileSizeL)(HFILE hFile, LONGLONG cbSize) = NULL;
ULONG (* _System __libc_gpfnDosSetFileLocksL)(HFILE hFile, __const__ FILELOCKL *pflUnlock, __const__ FILELOCKL *pflLock, ULONG ulTimeout, ULONG flFlags) = NULL;

/** Symlink EA name. */
#define EA_SYMLINK          "LIBC.SYMLINK"
/** Symlink EA owner. */
#define EA_UID              "LIBC.UID"
/** Symlink EA group. */
#define EA_GID              "LIBC.GID"
/** Symlink EA mode. */
#define EA_MODE             "LIBC.MODE"
/** Symlink EA i-node number. */
#define EA_INO              "LIBC.INO"
/** Symlink EA rdev number. */
#define EA_RDEV             "LIBC.RDEV"
/** Symlink EA gen number. */
#define EA_GEN              "LIBC.GEN"

/** Symlink EA name. */
static const char __libc_gszSymlinkEA[] = EA_SYMLINK;
/** UID EA name. */
static const char __libc_gszUidEA[]     = EA_UID;
/** GID EA name. */
static const char __libc_gszGidEA[]     = EA_GID;
/** Mode EA name. */
static const char __libc_gszModeEA[]    = EA_MODE;
/** Ino EA name. */
static const char __libc_gszInoEA[]     = EA_INO;
/** RDev EA name. */
static const char __libc_gszRDevEA[]    = EA_RDEV;
/** Gen(eration) EA name. */
static const char __libc_gszGenEA[]     = EA_GEN;

/* GEA2LIST Structure, just a bit more conveniently laid out.. */
static const struct
{
    ULONG   cbList;
    ULONG   oNextEntryOffset;
    BYTE    cbName;
    CHAR    szName[sizeof(EA_SYMLINK)];
} gGEA2ListSymlink =
{
    sizeof(gGEA2ListSymlink),
    0,
    sizeof(EA_SYMLINK) - 1,
    EA_SYMLINK
};

/* GEA2LIST Structures, just a bit more conveniently laid out.. */
static const struct GEA2LISTUNIXATTRIBS
{
    ULONG   cbList;

    ULONG   offSymlink;
    BYTE    cbSymlinkName;
    CHAR    szSymlinkName[sizeof(EA_SYMLINK)];

    ULONG   offUID;
    BYTE    cbUIDName;
    CHAR    szUIDName[sizeof(EA_UID)];

    ULONG   offGID;
    BYTE    cbGIDName;
    CHAR    szGIDName[sizeof(EA_GID)];

    ULONG   offMode;
    BYTE    cbModeName;
    CHAR    szModeName[sizeof(EA_MODE)];

    ULONG   offINO;
    BYTE    cbINOName;
    CHAR    szINOName[sizeof(EA_INO)];

    ULONG   offRDev;
    BYTE    cbRDevName;
    CHAR    szRDevName[sizeof(EA_RDEV)];

    ULONG   offGen;
    BYTE    cbGenName;
    CHAR    szGenName[sizeof(EA_GEN)];
} gGEA2ListUnixAttribs =
{
    sizeof(gGEA2ListUnixAttribs),
    offsetof(struct GEA2LISTUNIXATTRIBS, offUID)  - offsetof(struct GEA2LISTUNIXATTRIBS, offSymlink), sizeof(EA_SYMLINK) - 1, EA_SYMLINK,
    offsetof(struct GEA2LISTUNIXATTRIBS, offGID)  - offsetof(struct GEA2LISTUNIXATTRIBS, offUID),     sizeof(EA_UID) - 1,     EA_UID,
    offsetof(struct GEA2LISTUNIXATTRIBS, offMode) - offsetof(struct GEA2LISTUNIXATTRIBS, offGID),     sizeof(EA_GID) - 1,     EA_GID,
    offsetof(struct GEA2LISTUNIXATTRIBS, offINO)  - offsetof(struct GEA2LISTUNIXATTRIBS, offMode),    sizeof(EA_MODE) - 1,    EA_MODE,
    offsetof(struct GEA2LISTUNIXATTRIBS, offRDev) - offsetof(struct GEA2LISTUNIXATTRIBS, offINO),     sizeof(EA_INO) - 1,     EA_INO,
    offsetof(struct GEA2LISTUNIXATTRIBS, offGen)  - offsetof(struct GEA2LISTUNIXATTRIBS, offRDev),    sizeof(EA_RDEV) - 1,    EA_RDEV,
    0,                                                                                                sizeof(EA_GEN) - 1,     EA_GEN
};

/**
 * The special /@unixroot rewrite rule.
 */
static __LIBC_PATHREWRITE   gaRewriteRule =
{
    __LIBC_PRWF_CASE_SENSITIVE | __LIBC_PRWF_TYPE_DIR,      "/@unixroot",  10, __libc_gszUnixRoot, 0
};



/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int fsResolveUnix(const char *pszUserPath, unsigned fFlags, char *pszNativePath, int *pfInUnixTree);



/**
 * Init the file system stuff.
 *
 * @returns 0 on success.
 * @returns -1 on failure.
 */
int __libc_back_fsInit(void)
{
    LIBCLOG_ENTER("\n");

    /*
     * Only once.
     */
    static int  fInited;
    if (fInited)
        LIBCLOG_RETURN_INT(0);
    fInited = 1;

    /*
     * Create the mutex.
     */
    int rc = _fmutex_create(&__libc_gmtxFS, 0);
    if (rc)
        LIBCLOG_RETURN_INT(-1);

    /*
     * Inherit File System Data from parent.
     */
    __LIBC_PSPMINHERIT  pInherit = __libc_spmInheritRequest();
    __LIBC_PSPMINHFS    pFS;
    if (    pInherit
        &&  pInherit->cb > offsetof(__LIBC_SPMINHERIT, pFS)
        &&  (pFS = pInherit->pFS) != NULL
        &&  pFS->cchUnixRoot)
    {
        LIBCLOG_MSG("Inherited unixroot=%s fInUnixTree=%d cchUnixRoot=%d\n", pFS->szUnixRoot, pFS->fInUnixTree, pFS->cchUnixRoot);
        __libc_gfNoUnix     = 0;
        __libc_gfInUnixTree = pFS->fInUnixTree;
        __libc_gcchUnixRoot = pFS->cchUnixRoot;
        memcpy(&__libc_gszUnixRoot[0], &pFS->szUnixRoot[0], pFS->cchUnixRoot + 1);

        __libc_spmInheritRelease();

        /* Register rewrite rule. */
        gaRewriteRule.pszTo = "/";
        gaRewriteRule.cchTo = 1;
        if (__libc_PathRewriteAdd(&gaRewriteRule, 1))
            LIBCLOG_RETURN_INT(-1);
    }
    else
    {
        /*
         * Nothing to inherit.
         */
        if (pInherit)
            __libc_spmInheritRelease();
        if (!__libc_gfNoUnix)
        {
            /*
             * Setup unofficial unixroot.
             */
            const char *psz = getenv("UNIXROOT");
            if (psz)
            {
                LIBCLOG_MSG("Unofficial unixroot=%s\n", psz);
                size_t cch = strlen(psz);
                if (cch >= PATH_MAX - 32)
                {
                    LIBC_ASSERTM_FAILED("The UNIXROOT environment variable is too long! cch=%d maxlength=%d\n", cch, PATH_MAX - 32);
                    LIBCLOG_RETURN_INT(-1);
                }
                memcpy(&__libc_gszUnixRoot[0], psz, cch + 1);

                /* Register the rewrite rule. */
                gaRewriteRule.cchTo = cch;
                if (__libc_PathRewriteAdd(&gaRewriteRule, 1))
                    LIBCLOG_RETURN_INT(-1);
            }
        }
    }


    /*
     * Try resolve large file APIs.
     *
     * First, get a handle to DOSCALLS/DOSCALL1 which  is suitable
     * for DosQueryProcAddr.
     * Second, query the procedure addresses.
     * Third, free DOSCALLS since there's no point messing up any
     * reference counters for that module.
     */
    HMODULE     hmod = NULLHANDLE;
    if (DosLoadModule(NULL, 0, (PCSZ)"DOSCALLS", &hmod))
    {
        LIBC_ASSERTM_FAILED("DosLoadModule failed on doscalls!\n");
        abort();
        LIBCLOG_RETURN_INT(-1);
    }
    if (    DosQueryProcAddr(hmod, ORD_DOS32OPENL,          NULL, (PFN*)(void *)&__libc_gpfnDosOpenL)
        ||  DosQueryProcAddr(hmod, ORD_DOS32SETFILEPTRL,    NULL, (PFN*)(void *)&__libc_gpfnDosSetFilePtrL)
        ||  DosQueryProcAddr(hmod, ORD_DOS32SETFILESIZEL,   NULL, (PFN*)(void *)&__libc_gpfnDosSetFileSizeL)
        ||  DosQueryProcAddr(hmod, ORD_DOS32SETFILELOCKSL,  NULL, (PFN*)(void *)&__libc_gpfnDosSetFileLocksL)
          )
    {
        __libc_gpfnDosOpenL         = NULL;
        __libc_gpfnDosSetFilePtrL   = NULL;
        __libc_gpfnDosSetFileSizeL  = NULL;
        __libc_gpfnDosSetFileLocksL = NULL;
    }
    DosFreeModule(hmod);

    LIBCLOG_RETURN_INT(0);
}

/**
 * Pack inherit data.
 * @returns 0 on success.
 * @returns -1 on failure.
 * @param   ppFS    Where to store the pointer to the inherit data.
 * @param   pcbFS   Where to store the size of the inherit data.
 */
int __libc_back_fsInheritPack(__LIBC_PSPMINHFS *ppFS, size_t *pcbFS)
{
    LIBCLOG_ENTER("ppFS=%p pcbFS=%p\n", (void *)ppFS, (void *)pcbFS);

    *ppFS = NULL;
    *pcbFS = 0;

    if (__libc_back_fsMutexRequest())
        return -1;

    int rc = 0;
    if (__libc_gcchUnixRoot)
    {
        size_t cb = sizeof(__LIBC_SPMINHFS) + __libc_gcchUnixRoot;
        __LIBC_PSPMINHFS pFS = (__LIBC_PSPMINHFS)malloc(cb);
        if (pFS)
        {
            pFS->fInUnixTree = __libc_gfInUnixTree;
            pFS->cchUnixRoot = __libc_gcchUnixRoot;
            memcpy(pFS->szUnixRoot, __libc_gszUnixRoot, pFS->cchUnixRoot + 1);
            LIBCLOG_MSG("unixroot=%s fInUnixTree=%d cchUnixRoot=%d\n", pFS->szUnixRoot, pFS->fInUnixTree, pFS->cchUnixRoot);

            *pcbFS = cb;
            *ppFS = pFS;
        }
        else
            rc = -1;
    }

    __libc_back_fsMutexRelease();
    LIBCLOG_RETURN_INT(rc);
}


int __libc_back_fsMutexRequest(void)
{
    int rc = _fmutex_request(&__libc_gmtxFS, 0);
    if (!rc)
        return 0;
    return -__libc_native2errno(rc);
}


void __libc_back_fsMutexRelease(void)
{
    int rc = _fmutex_release(&__libc_gmtxFS);
    if (!rc)
        return;
    LIBC_ASSERTM_FAILED("_fmutex_release(gmtxFS) -> rc=%d\n", rc);
}


/**
 * Reads the content of a symbolic link.
 *
 * This is weird interface as it will return a truncated result if not
 * enough buffer space. It is also weird in that there is no string
 * terminator.
 *
 * @returns Number of bytes returned in pachBuf.
 * @returns -1 and errno on failure.
 * @param   pszNativePath   The path to the symlink to read.
 * @param   pachBuf         Where to store the symlink value.
 * @param   cchBuf          Size of buffer.
 */
int __libc_back_fsNativeSymlinkRead(const char *pszNativePath, char *pachBuf, size_t cchBuf)
{
    LIBCLOG_ENTER("pszNativePath=%p:{%s} pachBuf=%p cchBuf=%d\n", (void *)pszNativePath, pszNativePath, (void *)pachBuf, cchBuf);

    /*
     * Query the symlink EA value.
     */
    char    _achBuffer[SIZEOF_ACHBUFFER + 4];
    char   *pachBuffer = (char *)((uintptr_t)&_achBuffer[3] & ~3);
    EAOP2   EaOp;
    EaOp.fpGEA2List = (PGEA2LIST)&gGEA2ListSymlink;
    EaOp.fpFEA2List = (PFEA2LIST)pachBuffer;
    EaOp.oError     = 0;
    EaOp.fpFEA2List->cbList = SIZEOF_ACHBUFFER;
    int rc = DosQueryPathInfo((PCSZ)pszNativePath, FIL_QUERYEASFROMLIST, &EaOp, sizeof(EaOp));
    if (!rc)
    {
        if (    EaOp.fpFEA2List->cbList > sizeof(EaOp.fpFEA2List->list[0])
            &&  EaOp.fpFEA2List->list[0].cbValue)
        {
            /*
             * Validate the EA.
             */
            PUSHORT pusType = (PUSHORT)((char *)&EaOp.fpFEA2List->list[1] + EaOp.fpFEA2List->list[0].cbName);
            int     cchSymlink = pusType[1];
            char   *pszSymlink = (char *)&pusType[2];
            if (    pusType[0] == EAT_ASCII
                &&  cchSymlink < EaOp.fpFEA2List->list[0].cbValue
                &&  cchSymlink > 0
                &&  *pszSymlink)
            {
                /*
                 * Copy it to the buffer and return.
                 */
                if (cchSymlink > cchBuf)
                {
                    LIBC_ASSERTM_FAILED("Buffer to small. %d bytes required, %d bytes available. pszSymlink='%.*s' pszNativePath='%s'\n",
                                        cchSymlink, cchBuf, cchSymlink, pszSymlink, pszNativePath);
                    cchSymlink = cchBuf;
                }
                memcpy(pachBuf, pszSymlink, cchSymlink);
                LIBCLOG_RETURN_INT(cchSymlink);
            }
            else
            {
                LIBC_ASSERTM_FAILED("Invalid symlink EA! type=%x len=%d cbValue=%d *pszSymlink=%c pszNativePath='%s'\n",
                                    pusType[0], pusType[1], EaOp.fpFEA2List->list[0].cbValue, *pszSymlink, pszNativePath);
                LIBCLOG_RETURN_INT(-EFTYPE);
            }
        }
        else
        {
            LIBC_ASSERTM_FAILED("%s is not a symlink!\n", pszNativePath);
            LIBCLOG_RETURN_INT(-EINVAL);
        }
    }
    else
        LIBC_ASSERTM_FAILED("DosQueryPathInfo(%s) -> %d!\n", pszNativePath, rc);

    /* failure */
    rc = -__libc_native2errno(rc);
    LIBCLOG_RETURN_INT(rc);
}

#if 0 //just testing, not useful.
/**
 * Updates a symlink.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszTarget       The symlink target.
 * @param   pszSymlink      The symlink, the one to be updated.
 */
int __libc_back_fsSymlinkWrite(const char *pszTarget, const char *pszSymlink)
{
    LIBCLOG_ENTER("pszTarget=%s pszSymlink=%s\n", pszTarget, pszSymlink);

    /*
     * Validate input.
     */
    int cchTarget = strlen(pszTarget);
    if (cchTarget >= PATH_MAX)
    {
        errno = ENAMETOOLONG;
        LIBC_ASSERTM_FAILED("Target is too long, %d bytes. (%s)\n", cchTarget, pszTarget);
        LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Allocate FEA buffer.
     */
    EAOP2   EaOp = {0,0,0};
    int     cb = cchTarget + sizeof(USHORT) * 2 + sizeof(EA_SYMLINK) + sizeof(FEALIST);
    EaOp.fpFEA2List = alloca(cb);
    if (!EaOp.fpFEA2List)
    {
        LIBC_ASSERTM_FAILED("Out of stack! alloca(%d) failed\n", cb);
        LIBCLOG_RETURN_INT(-ENOMEM);
    }

    /*
     * Setup the EAOP structure.
     */
    EaOp.fpFEA2List->cbList = cb;
    PFEA2   pFEA = &EaOp.fpFEA2List->list[0];
    pFEA->oNextEntryOffset  = 0;
    pFEA->cbName            = sizeof(EA_SYMLINK) - 1;
    pFEA->cbValue           = cchTarget + sizeof(USHORT) * 2;
    pFEA->fEA               = FEA_NEEDEA;
    memcpy(pFEA->szName, EA_SYMLINK, sizeof(EA_SYMLINK));
    PUSHORT pus = (PUSHORT)&pFEA->szName[sizeof(EA_SYMLINK)];
    pus[0]                  = EAT_ASCII;
    pus[1]                  = cchTarget;
    memcpy(&pus[2], pszTarget, cchTarget);
    int rc = DosSetPathInfo((PCSZ)pszTarget, FIL_QUERYEASIZE, &EaOp, sizeof(EAOP2), 0);
    if (!rc)
        LIBCLOG_RETURN_INT(0);

    rc = -__libc_native2errno(rc);
    LIBCLOG_RETURN_INT(rc);
}
#endif


/**
 * Creates a symlink.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure.
 * @param   pszTarget       The symlink target.
 * @param   pszSymlink      The symlink, the one to be updated.
 */
int __libc_back_fsNativeSymlinkCreate(const char *pszTarget, const char *pszNativePath)
{
    LIBCLOG_ENTER("pszTarget=%s pszNativePath=%s\n", pszTarget, pszNativePath);

    /*
     * Validate input.
     */
    int cchTarget = strlen(pszTarget);
    if (cchTarget >= PATH_MAX)
    {
        LIBC_ASSERTM_FAILED("Target is too long, %d bytes. (%s)\n", cchTarget, pszTarget);
        LIBCLOG_RETURN_INT(-ENAMETOOLONG);
    }

    /*
     * Allocate FEA buffer.
     */
    EAOP2   EaOp = {0,0,0};
    int     cb = cchTarget + sizeof(USHORT) * 2 + sizeof(EA_SYMLINK) + sizeof(FEALIST);
    EaOp.fpFEA2List = alloca(cb);
    if (!EaOp.fpFEA2List)
    {
        LIBC_ASSERTM_FAILED("Out of stack! alloca(%d) failed\n", cb);
        LIBCLOG_RETURN_INT(-ENOMEM);
    }

    /*
     * Setup the EAOP structure.
     */
    EaOp.fpFEA2List->cbList = cb;
    PFEA2   pFEA = &EaOp.fpFEA2List->list[0];
    pFEA->oNextEntryOffset  = 0;
    pFEA->cbName            = sizeof(EA_SYMLINK) - 1;
    pFEA->cbValue           = cchTarget + sizeof(USHORT) * 2;
    pFEA->fEA               = FEA_NEEDEA;
    memcpy(pFEA->szName, EA_SYMLINK, sizeof(EA_SYMLINK));
    PUSHORT pus = (PUSHORT)&pFEA->szName[sizeof(EA_SYMLINK)];
    pus[0]                  = EAT_ASCII;
    pus[1]                  = cchTarget;
    memcpy(&pus[2], pszTarget, cchTarget);
    HFILE   hf = -1;
    ULONG   ul = 0;
    int rc = DosOpen((PCSZ)pszNativePath, &hf, &ul, cchTarget + 1, FILE_NORMAL,
                     OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_FAIL_IF_EXISTS,
                     OPEN_FLAGS_NOINHERIT | OPEN_SHARE_DENYNONE | OPEN_ACCESS_READWRITE,
                     &EaOp);
    if (!rc)
    {
        char *psz = (char *)&pus[2] - (sizeof(EA_SYMLINK "=") - 1);
        memcpy(psz, EA_SYMLINK "=", sizeof(EA_SYMLINK "=") - 1);
        DosWrite(hf, psz, cchTarget + sizeof(EA_SYMLINK "=") - 1, &ul);
        DosClose(hf);
        LIBCLOG_RETURN_INT(0);
    }

    rc = -__libc_native2errno(rc);
    LIBCLOG_RETURN_INT(rc);
}



/**
 * Updates the global unix root stuff.
 * Assumes caller have locked the fs stuff.
 *
 * @param   pszUnixRoot     The new unix root. Fully resolved and existing.
 */
void __libc_back_fsUpdateUnixRoot(const char *pszUnixRoot)
{
    gaRewriteRule.pszTo = "/";
    gaRewriteRule.cchTo = 1;

    int cch = strlen(pszUnixRoot);
    memcpy(__libc_gszUnixRoot, pszUnixRoot, cch + 1);
    __libc_gcchUnixRoot = cch;
}


/**
 * Cleans up a path specifier a little bit.
 * This includes removing duplicate slashes, uncessary single dots, and
 * trailing slashes.
 *
 * @returns Number of bytes in the clean path.
 * @param   pszPath     The path to cleanup.
 */
static int fsCleanPath(char *pszPath)
{
    /*
     * Change to '/' and remove duplicates.
     */
    int     fUnc = 0;
    char   *pszSrc = pszPath;
    char   *pszTrg = pszPath;
    if (    (pszPath[0] == '\\' || pszPath[0] == '/')
        &&  (pszPath[1] == '\\' || pszPath[1] == '/'))
    {   /* Skip first slash in a unc path. */
        pszSrc++;
        *pszTrg++ = '/';
        fUnc = 1;
    }

    for (;;)
    {
        char ch = *pszSrc++;
        if (ch == '/' || ch == '\\')
        {
            *pszTrg++ = '/';
            for (;;)
            {
                do  ch = *pszSrc++;
                while (ch == '/' || ch == '\\');

                /* Remove '/./' and '/.'. */
                if (ch != '.' || (*pszSrc && *pszSrc != '/' && *pszSrc != '\\'))
                    break;
            }
        }
        *pszTrg = ch;
        if (!ch)
            break;
        pszTrg++;
    }

    /*
     * Remove trailing slash.
     */
    int cch = pszTrg - pszPath;
    if (cch > 1 && pszTrg[-1] == '/' && pszTrg[-2] != ':' && pszTrg[-2] != '/')
        pszPath[--cch] = '\0';

    return cch;
}


/**
 * Resolves and verifies the user path to a native path.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszUserPath     The user path.
 * @parm    fFlags          Flags controlling the operation of the function.
 *                          See the BACKFS_FLAGS_* defines.
 * @param   pszNativePath   Where to store the native path. This buffer is at
 *                          least PATH_MAX bytes big.
 * @param   pfInUnixTree    Where to store the result-in-unix-tree indicator. Optional.
 */
int __libc_back_fsResolve(const char *pszUserPath, unsigned fFlags, char *pszNativePath, int *pfInUnixTree)
{
    if (!__libc_gfNoUnix)
        return fsResolveUnix(pszUserPath, fFlags, pszNativePath, pfInUnixTree);

    /* __libc_back_fsResolveOS2(): */

    LIBCLOG_ENTER("pszUserPath=%p:{%s} fFlags=%x pszNativePath=%p pfInUnixTree=%p\n",
                  (void *)pszUserPath, pszUserPath, fFlags, (void *)pszNativePath, (void *)pfInUnixTree);

    /*
     * Apply rewrite rule.
     */
    int cch = __libc_PathRewrite(pszUserPath, pszNativePath, sizeof(pszNativePath));
    if (cch < 0)
        LIBCLOG_RETURN_INT(-EINVAL);
    if (cch == 0)
    {
        cch = strlen(pszUserPath);
        if (cch >= PATH_MAX)
            LIBCLOG_RETURN_INT(-ENAMETOOLONG);
        memcpy(pszNativePath, pszUserPath, cch + 1);
    }

    /*
     * Convert slashes.
     */
    char *psz = strchr(pszNativePath, '/');
    while (psz)
    {
        *psz++ = '\\';
        psz = strchr(psz, '/');
    }

    /** @todo Validate the path? hopefully not necessary. */

    if (pfInUnixTree)
        *pfInUnixTree = 0;

    LIBCLOG_RETURN_INT(0);
}


/**
 * Resolves and verifies the user path to a native path.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszUserPath     The user path.
 * @parm    fFlags          Flags controlling the operation of the function.
 *                          See the BACKFS_FLAGS_* defines.
 * @param   pszNativePath   Where to store the native path. This buffer is at
 *                          least PATH_MAX bytes big.
 * @param   pfInUnixTree    Where to store the result-in-unix-tree indicator. Optional.
 */
int fsResolveUnix(const char *pszUserPath, unsigned fFlags, char *pszNativePath, int *pfInUnixTree)
{
    LIBCLOG_ENTER("pszUserPath=%p:{%s} pszNativePath=%p *pfInUnixTree=%p\n",
                  (void *)pszUserPath, pszUserPath, (void *)pszNativePath, (void *)pfInUnixTree);
    const char *pszUserPathIn = pszUserPath;
    char        szTmp[PATH_MAX];
    char        _achBuffer[SIZEOF_ACHBUFFER + 4];
    char       *pachBuffer = (char *)((uintptr_t)&_achBuffer[3] & ~3);
    unsigned    cLoopsLeft = 8;
    int         fInUnixTree = __libc_gfInUnixTree;
    int         rcRet = -ENAMETOOLONG;
    HDIR        hDir = HDIR_CREATE;
    FS_VAR()
    FS_SAVE_LOAD();
    while (cLoopsLeft-- > 0)
    {
        /*
         * Determin root slash position.
         */
        int iRoot;
        if (pszUserPath[0] == '/' || pszUserPath[0] == '\\')
        {
            iRoot = 0;
            fInUnixTree = __libc_gfInUnixTree;
        }
        else if (pszUserPath[0] && pszUserPath[1] == ':'
                 && (pszUserPath[2] == '/' || pszUserPath[2] == '\\'))
        {
            iRoot = 2;
            fInUnixTree = 0;
        }
        /*
         * No root slash? Make one!
         * This can only happen in the first pass (unless something BAD happens of course).
         */
        else
        {
            ULONG   ul;
            int     rc;
            if (pszUserPath[1] != ':')
            {
                /*
                 * Current drive.
                 */
                ULONG ulDisk;
                rc = DosQueryCurrentDisk(&ulDisk, &ul);
                szTmp[0] = ulDisk + 'A' - 1;
                ul = sizeof(szTmp) - 2;
                if (!rc)
                    rc = DosQueryCurrentDir(0, (PSZ)&szTmp[3], &ul);
                iRoot = __libc_gfInUnixTree ? __libc_gcchUnixRoot : 2;
                /* fInUnixTree remains whatever it is */
            }
            else
            {
                /*
                 * Drive letter but no root slash.
                 */
                szTmp[0] = pszUserPath[0];
                ul = sizeof(szTmp) - 2;
                rc = DosQueryCurrentDir(pszUserPath[0] - (pszUserPath[0] >= 'A' && pszUserPath[0] <= 'Z' ? 'A' : 'a') + 1, (PSZ)&szTmp[3], &ul);
                pszUserPath += 2;
                iRoot = 2;
                fInUnixTree = 0;
            }
            /* failed? */
            if (rc)
            {
                rcRet = -__libc_native2errno(rc);
                break;
            }

            /*
             * Add the path stuff from the user.
             */
            szTmp[1] = ':';
            szTmp[2] = '/';
            int cch = strlen(szTmp);
            szTmp[cch++] = '/';
            int cchUserPath = strlen(pszUserPath) + 1;
            if (cch + cchUserPath > PATH_MAX)
                break;
            memcpy(&szTmp[cch], pszUserPath, cchUserPath);
            pszUserPath = memcpy(pachBuffer, szTmp, cch + cchUserPath);
        }

        /*
         * Verify any drive specifier.
         */
        if (pszUserPath[1] == ':')
        {
            if (*pszUserPath >= 'a' && *pszUserPath <= 'z')
            {
                if ((uintptr_t)(pszUserPath - pachBuffer) > SIZEOF_ACHBUFFER)
                    pszUserPath = strcpy(pachBuffer, pszUserPath);
                *(char *)(void *)pszUserPath += 'A' - 'a';
            }
            else if (!(*pszUserPath >= 'A' && *pszUserPath <= 'Z'))
            {
                rcRet = -ENOENT;
                break;
            }
        }

        /*
         * Apply rewrite rules.
         * Path now goes to szTmp buffer.
         */
        int cchTmp = __libc_PathRewrite(pszUserPath, szTmp, sizeof(szTmp));
        if (cchTmp > 0)
        {
            /*
             * Redetermin root because of rewrite.
             * Assume qualified path from rewrite!
             */
            iRoot = szTmp[1] == ':' ? 2 : 1;
        }
        else if (!cchTmp)
        {
            cchTmp = strlen(pszUserPath);
            if (cchTmp + 2 > sizeof(szTmp))
                break;
            memcpy(szTmp, pszUserPath, cchTmp + 1);
        }
        else
        {
            rcRet = -EINVAL;
            break;
        }

        /*
         * Check if special OS/2 name.
         */
        if (pszUserPath[0] == '/' || pszUserPath[0] == '\\')
        {
            int fDone = 0;
            if (    (pszUserPath[1] == 'p' || pszUserPath[1] == 'P')
                &&  (pszUserPath[2] == 'i' || pszUserPath[2] == 'I')
                &&  (pszUserPath[3] == 'p' || pszUserPath[3] == 'P')
                &&  (pszUserPath[4] == 'e' || pszUserPath[4] == 'E')
                &&  (pszUserPath[5] == '/' || pszUserPath[5] == '\\'))
                fDone = 1;
            else if ((pszUserPath[1]== 'd' || pszUserPath[1] == 'D')
                &&  (pszUserPath[2] == 'e' || pszUserPath[2] == 'E')
                &&  (pszUserPath[3] == 'v' || pszUserPath[3] == 'V')
                &&  (pszUserPath[4] == '/' || pszUserPath[4] == '\\')
                )
            {
                PFSQBUFFER2 pfsqb = (PFSQBUFFER2)&szTmp[0];
                ULONG       cb = sizeof(szTmp);
                fDone = !DosQueryFSAttach((PCSZ)pszUserPath, 0, FSAIL_QUERYNAME, pfsqb, &cb);
            }

            /* If it was, copy path to szTmp and go to exit code. */
            if (fDone)
            {
                int cch = strlen(szTmp) + 1;
                if (cch <= PATH_MAX)
                {
                    memcpy(szTmp, pszUserPath, cchTmp + 1);
                    fsCleanPath(&szTmp[0]);
                    rcRet = 0;
                }
                break;
            }
        }

        /*
         * Remove excessive slashing and convert all slashes to '/'.
         */
        cchTmp = fsCleanPath(&szTmp[0]);

        /*
         * Expand unix root.
         */
        if (szTmp[0] == '/' && szTmp[1] != '/' && __libc_gcchUnixRoot)
        {
            iRoot = __libc_gcchUnixRoot;
            if (cchTmp + iRoot >= PATH_MAX)
                break;
            memcpy(pachBuffer, szTmp, cchTmp + 1);
            memcpy(szTmp, __libc_gszUnixRoot, iRoot);
            if (cchTmp != 1 || iRoot <= 2)
                memcpy(&szTmp[iRoot], pachBuffer, cchTmp + 1);
            else
                szTmp[iRoot] = '\0';
            cchTmp += iRoot;

            fInUnixTree = 1;
        }


        /*
         * Check all directory components.
         *
         * We actually don't bother checking if they're directories, only that
         * they exists. This shouldn't matter much since any operation on the
         * next/final component will assume the previous one being a directory.
         *
         * While walking we will clean up all use of '.' and '..' so we'll end
         * up with an optimal path in the end.
         */

        /*
         * Find the end of the first component (start and end + 1).
         */
        char *pszPrev;
        char *psz;
        if (szTmp[0] == '/' && szTmp[1] == '/')
        {
            /* UNC - skip past the server name. */
            psz = strchr(&szTmp[2], '/');
            if (!psz)
            {
                rcRet = -ENOENT;
                break;
            }
            pszPrev = ++psz;
            iRoot = psz - &szTmp[0];    /* Unix root cannot be UNC. */
            psz = strchr(psz, '/');
        }
        else
        {
            pszPrev = &szTmp[iRoot + 1];
            psz = strchr(pszPrev, '/');
        }
        LIBC_ASSERTM(pszPrev - &szTmp[0] >= iRoot, "iRoot=%d  pszPrev offset %d  szTmp=%s\n", iRoot, pszPrev - &szTmp[0], szTmp);
        /* If only one component, we'll check if the fVerifyLast was requested. */
        if (    !psz
            &&  (fFlags == BACKFS_FLAGS_RESOLVE_FULL || fFlags == BACKFS_FLAGS_RESOLVE_FULL_MAYBE))
            psz = strchr(szTmp, '\0');

        /*
         * Walking loop.
         */
        int fDone = 1;
        while (psz)
        {
            char chSlash = *psz;
            *psz = '\0';

            /*
             * Kill . and .. specs.
             */
            if (    pszPrev[0] == '.'
                &&  (   pszPrev[1] == '\0'
                     || (pszPrev[1] == '.' && pszPrev[2] == '\0') ) )
            {
                if (    pszPrev[1] != '\0'
                    &&  (uintptr_t)(pszPrev - &szTmp[0]) != iRoot + 1)
                {
                    for (pszPrev -= 2; *pszPrev != '/'; pszPrev--)
                        /* noop */;
                    pszPrev++;
                }
                *psz = chSlash;
                memmove(pszPrev - 1, psz, cchTmp - (psz - &szTmp[0]) + 1);
                cchTmp -= psz - pszPrev - 1;

                /*
                 * Next path component and restart loop.
                 */
                if (!chSlash)
                {
                    rcRet = -0;
                    break;
                }
                psz = pszPrev;
                while (*psz != '/')
                {
                    if (!*psz)
                    {
                        if (    fFlags != BACKFS_FLAGS_RESOLVE_FULL
                            &&  fFlags != BACKFS_FLAGS_RESOLVE_FULL_MAYBE)
                        {
                            rcRet = -0;
                            psz = NULL;
                        }
                        break;
                    }
                    psz++;
                }
                continue;
            }

            /*
             * Check for obviously illegal characters in this path component
             * saving us from having DosFindFirst getting '*' and '?'.
             */
            if (strpbrk(pszPrev, "*?"))
            {
                rcRet = -ENOENT;        /* hmm. will this be correct for all situation? */
                break;
            }

            /*
             * Find the correct name and to check if it is a directory or not.
             * This'll of course also provide proper verification of the path too. :-)
             */
            PFILEFINDBUF4 pFindBuf = (PFILEFINDBUF4)pachBuffer;
            ULONG cFiles = 1;
            int rc = DosFindFirst((PCSZ)&szTmp[0], &hDir, FILE_READONLY | FILE_HIDDEN | FILE_SYSTEM | FILE_DIRECTORY | FILE_ARCHIVED,
                                  pFindBuf, SIZEOF_ACHBUFFER, &cFiles, FIL_QUERYEASIZE);
            if (rc || cFiles == 0)
            {
                LIBCLOG_MSG("DosFindFirst('%s',,,,,) -> %d resolving '%s'\n", szTmp, rc, pszUserPathIn);
                if (fFlags == BACKFS_FLAGS_RESOLVE_FULL_MAYBE && !chSlash)
                    rcRet = 0;
                else
                    rcRet = rc == ERROR_FILE_NOT_FOUND && chSlash ? -ENOENT : -__libc_native2errno(rc);
                hDir = HDIR_CREATE;
                break;
            }
            memcpy(pszPrev, pFindBuf->achName, psz - pszPrev);
            int     fIsDirectory = (pFindBuf->attrFile & FILE_DIRECTORY) != 0;

            /*
             * Try querying the symlink EA value.
             * (This operation will reuse the achBuffer overwriting the pFindBuf data!)
             *
             * Yeah, we could do this in the same operation as the DosFindFirst() but
             * a little bird told me that level 3 DosFindFirst request had not been
             * returning the right things at some point, and besides it's return data
             * is rather clumsily laid out. So, I decided not to try my luck on it.
             */
            if (pFindBuf->cbList > sizeof(USHORT) * 2 + 1)
            {
                EAOP2   EaOp;
                EaOp.fpGEA2List = (PGEA2LIST)&gGEA2ListSymlink;
                EaOp.fpFEA2List = (PFEA2LIST)pachBuffer;
                EaOp.oError     = 0;
                EaOp.fpFEA2List->cbList = SIZEOF_ACHBUFFER;
                rc = DosQueryPathInfo((PCSZ)&szTmp[0], FIL_QUERYEASFROMLIST, &EaOp, sizeof(EaOp));
                if (rc)
                {
                    LIBCLOG_MSG("DosQueryPathInfo('%s',,,) -> %d resolving '%s'\n", szTmp, rc, pszUserPathIn);
                    if (fFlags == BACKFS_FLAGS_RESOLVE_FULL_MAYBE && !chSlash)
                        rcRet = 0;
                    else
                        rcRet = rc == ERROR_FILE_NOT_FOUND && chSlash ? -ENOENT : -__libc_native2errno(rc);
                    break;
                }

                /*
                 * Did we find any symlink EA?
                 */
                if (    EaOp.fpFEA2List->cbList > sizeof(EaOp.fpFEA2List->list[0])
                    &&  EaOp.fpFEA2List->list[0].cbValue)
                {
                    /* Validate the EA. */
                    PUSHORT pusType = (PUSHORT)((char *)&EaOp.fpFEA2List->list[1] + EaOp.fpFEA2List->list[0].cbName);
                    char   *pszSymlink = (char *)&pusType[2];
                    if (    pusType[0] != EAT_ASCII
                        ||  pusType[1] > EaOp.fpFEA2List->list[0].cbValue
                        ||  !pusType[1]
                        ||  !*pszSymlink)
                    {
                        LIBC_ASSERTM_FAILED("Invalid symlink EA! type=%x len=%d cbValue=%d *pszSymlink=%c\n",
                                            pusType[0], pusType[1], EaOp.fpFEA2List->list[0].cbValue, *pszSymlink);
                        rcRet = -EFTYPE;
                        break;
                    }

                    /* Cleanup the symlink and find it's length. */
                    pszSymlink[pusType[1]] = '\0';
                    int cchSymlink = fsCleanPath(pszSymlink);

                    /* Merge symlink with the path. */
                    int cchLeft = cchTmp - (psz - &szTmp[0]);
                    if (*pszSymlink == '/' || *pszSymlink == '\\' || pszSymlink[1] == ':')
                    {
                        /*
                         * Replace the path up to the current point with the symlink,
                         */
                        if (cchSymlink + cchLeft + 2 >= PATH_MAX)
                            break;
                        if (cchLeft)
                        {
                            if (pszSymlink[cchSymlink - 1] != '/')
                                pszSymlink[cchSymlink++] = '/';
                            memcpy(&pszSymlink[cchSymlink], psz + 1, cchLeft + 1);
                        }

                        /* restart the whole shebang. */
                        pszUserPath = pszSymlink;
                        fDone = 0;
                        break;
                    }
                    else
                    {
                        /*
                         * Squeeze the symlink in instead of the current path component.
                         */
                        if (cchSymlink + cchTmp + 2 >= PATH_MAX)
                            break;
                        if (cchLeft)
                        {
                            pszSymlink[cchSymlink++] = '/';
                            memcpy(&pszSymlink[cchSymlink], psz + 1, cchLeft + 1);
                            cchSymlink += cchLeft;
                        }
                        memcpy(pszPrev, pszSymlink, cchSymlink + 1);

                        psz = pszPrev;
                        while (*psz != '/')
                        {
                            if (!*psz)
                            {
                                if (    fFlags != BACKFS_FLAGS_RESOLVE_FULL
                                    &&  fFlags != BACKFS_FLAGS_RESOLVE_FULL_MAYBE)
                                {
                                    rcRet = 0;
                                    psz = NULL;
                                }
                                break;
                            }
                            psz++;
                        }
                        continue;
                    }
                }
            }

            /*
             * If we get here it was not a symlink and we should check if the directoryness
             * of it fit's into the big picture...
             */
            if (!fIsDirectory && chSlash)
            {
                LIBCLOG_MSG("'%s' is not a directory (resolving '%s')\n", szTmp, pszUserPathIn);
                rcRet = -ENOTDIR;
                break;
            }

            /*
             * Next path component.
             */
            *psz++ = chSlash;
            if (!chSlash)
            {
                rcRet = 0;
                break;
            }
            pszPrev = psz;
            while (*psz != '/')
            {
                if (!*psz)
                {
                    if (    fFlags != BACKFS_FLAGS_RESOLVE_FULL
                        &&  fFlags != BACKFS_FLAGS_RESOLVE_FULL_MAYBE)
                    {
                        rcRet = 0;
                        psz = NULL;
                    }
                    break;
                }
                psz++;
            }
        } /* inner loop */

        if (fDone)
            break;
    } /* outer loop */

    /*
     * Cleanup find handle.
     */
    if (hDir != HDIR_CREATE)
        DosFindClose(hDir);
    FS_RESTORE();

    /*
     * Copy the resolved path the the caller buffer.
     */
    if (!rcRet)
    {
        int cch = strlen(szTmp) + 1;
        if (cch == 1 || (cch == 3 && szTmp[1] == ':'))
        {
            szTmp[cch - 1] = '/';
            szTmp[cch++] = '\0';
        }
        memcpy(pszNativePath, szTmp, cch);
        if (pfInUnixTree)
            *pfInUnixTree = fInUnixTree;
        LIBCLOG_RETURN_INT(0);
    }

    /* failure */
    LIBCLOG_RETURN_INT(rcRet);
    pszUserPathIn = pszUserPathIn;
}


/**
 * Creates the unix EAs for a new file or directory.
 * @returns 0 on success.
 * @returns Negative errno on failure.
 * @param   hFile           File handle to the newly created fs object. If no
 *                          handle handy, set to -1.
 * @param   pszNativePath   Native path to the newly created fs object. If
 *                          handle is give this will be ignored.
 * @param   mode            The specified file permission mode mask.
 */
int __libc_back_fsUnixAttribsSet(int hFile, const char *pszNativePath, mode_t mode)
{
    /* later */
    return 0;
}

/**
 * Reads the unix EAs for a file which is being stat'ed.
 *
 * @returns 0 on success.
 * @returns Negative errno on failure.
 * @param   hFile           File handle to the fs object. If no handle handy, set to -1.
 * @param   pszNativePath   Native path to the fs object. If handle is give this will be ignored.
 * @param   pStat           Pointer to the stat buffer.
 *                          The buffer is only updated if and with the EAs we find,
 *                          so the caller must fill the fields with defaults before
 *                          calling this function.
 */
int __libc_back_fsUnixAttribsGet(int hFile, const char *pszNativePath, struct stat *pStat)
{
    LIBCLOG_ENTER("hFile=%d pszNativePath=%p:{%s} pStat=%p\n", hFile, (void *)pszNativePath, pszNativePath, (void *)pStat);

    /* Try come up with an accurate max estimate of a maximum result. */
    char    achBuffer[sizeof(gGEA2ListUnixAttribs) + 7 * (sizeof(USHORT) * 2 + sizeof(BYTE)) + CCHMAXPATH + 6 * sizeof(uint32_t) + 0x30];
    char   *pachBuffer = (char *)((uintptr_t)&achBuffer[3] & ~3);
    EAOP2   EaOp2;
    int     rc;

    /*
     * Issue the query.
     */
    EaOp2.fpGEA2List = (PGEA2LIST)&gGEA2ListUnixAttribs;
    EaOp2.fpFEA2List = (PFEA2LIST)pachBuffer;
    EaOp2.oError     = 0;
    EaOp2.fpFEA2List->cbList = sizeof(achBuffer) - 4;
    if (hFile >= 0)
        rc = DosQueryFileInfo(hFile, FIL_QUERYEASFROMLIST, &EaOp2, sizeof(EaOp2));
    else
        rc = DosQueryPathInfo((PCSZ)pszNativePath, FIL_QUERYEASFROMLIST, &EaOp2, sizeof(EaOp2));
    if (rc)
    {
        LIBC_ASSERTM_FAILED("Bogus EAs? rc=%d oError=%ld\n", rc, EaOp2.oError);
        rc = -__libc_native2errno(rc);
        LIBCLOG_RETURN_INT(rc);
    }
    if (EaOp2.fpFEA2List->cbList < LIBC_UNIX_EA_MIN)
        LIBCLOG_RETURN_INT(rc);

    /*
     * Parse the result.
     */
    PFEA2   pFea2 = &EaOp2.fpFEA2List->list[0];
    for (;;)
    {
        if (pFea2->cbValue > 0)
        {
#define COMPARE_EANAME(name) (pFea2->cbName == sizeof(name) - 1 && !memcmp(name, pFea2->szName, sizeof(name) - 1))
            if (COMPARE_EANAME(__libc_gszSymlinkEA))
                pStat->st_mode = (pStat->st_mode & ~S_IFMT) | S_IFLNK;
            else
            {
                PUSHORT pusType = (PUSHORT)((char *)pFea2 + pFea2->cbName);
                if (pusType[0] == EAT_BINARY && pusType[1] == sizeof(uint32_t))
                {
                    uint32_t u32 = *(uint32_t *)&pusType[2];
                    if (COMPARE_EANAME(__libc_gszUidEA))
                        pStat->st_uid  = u32;
                    else if (COMPARE_EANAME(__libc_gszGidEA))
                        pStat->st_gid  = u32;
                    else if (COMPARE_EANAME(__libc_gszModeEA))
                        pStat->st_mode = u32;
                    else if (COMPARE_EANAME(__libc_gszInoEA))
                        pStat->st_ino  = u32;
                    else if (COMPARE_EANAME(__libc_gszRDevEA))
                        pStat->st_rdev = u32;
                    else if (COMPARE_EANAME(__libc_gszGenEA))
                        /* pStat->st_gen  = u32 - not implemented */;
                    else
                        LIBC_ASSERTM_FAILED("Huh?!? got an ea named '%s', namelen=%d!\n", pFea2->szName, pFea2->cbName);
                }
                else
                    LIBC_ASSERTM_FAILED("Invalid LIBC EA! %s type=%#x len=%#x, expected type=%#x and len=%#x.\n",
                                        pFea2->szName, pusType[0], pusType[1], EAT_BINARY, sizeof(uint32_t));
            }
#undef COMPARE_EANAME
        }

        /* next */
        if (pFea2->oNextEntryOffset <= sizeof(FEA2))
            break;
        pFea2 = (PFEA2)((uintptr_t)pFea2 + pFea2->oNextEntryOffset);
    }

    LIBCLOG_RETURN_INT(0);
}


