/* $Id: fs.h 1519 2004-09-27 02:15:07Z bird $ */
/** @file
 *
 * LIBC SYS Backend - file system stuff.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird@innotek.de>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __fs_h__
#define __fs_h__

#include <sys/cdefs.h>
#include <sys/syslimits.h>
#include <sys/types.h>
#include <InnoTekLIBC/sharedpm.h>
#include "backend.h"

struct stat;

__BEGIN_DECLS

/** Indicator whether or not we're in the unix tree. */
extern int  __libc_gfInUnixTree;
/** Length of the unix root if the unix root is official. */
extern int  __libc_gcchUnixRoot;
/** The current unix root. */
extern char __libc_gszUnixRoot[PATH_MAX];
/** The current umask of the process. */
extern mode_t __libc_gfsUMask;

#ifdef _OS2EMX_H
/** @group Large File Support - API pointers.
 * Test if the function pointer is set before calling it.
 * For APIs which take a structure level parameter, check __pfnDosOpenL.
 * @{ */
extern ULONG (* _System __libc_gpfnDosOpenL)(PCSZ pszFileName, PHFILE phFile, PULONG pulAction, LONGLONG llFileSize, ULONG ulAttribute, ULONG ulOpenFlags, ULONG ulOpenMode, PEAOP2 pEABuf);
extern ULONG (* _System __libc_gpfnDosSetFilePtrL)(HFILE hFile, LONGLONG llOffset, ULONG ulOrigin, PLONGLONG pllPos);
extern ULONG (* _System __libc_gpfnDosSetFileSizeL)(HFILE hFile, LONGLONG cbSize);
extern ULONG (* _System __libc_gpfnDosSetFileLocksL)(HFILE hFile, __const__ FILELOCKL *pflUnlock, __const__ FILELOCKL *pflLock, ULONG ulTimeout, ULONG flFlags);
/** @} */
#endif

/**
 * Init the file system stuff.
 *
 * @returns 0 on success.
 * @returns -1 on failure.
 */
int __libc_back_fsInit(void);

/**
 * Pack inherit data.
 * @returns 0 on success.
 * @returns -1 on failure.
 * @param   ppFS    Where to store the pointer to the inherit data.
 * @param   pcbFS   Where to store the size of the inherit data.
 */
int __libc_back_fsInheritPack(__LIBC_PSPMINHFS *ppFS, size_t *pcbFS);

/**
 * Request the owner ship of the FS mutex.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 */
int __libc_back_fsMutexRequest(void);

/**
 * Releases the owner ship of the FS mutex obtained by __libc_back_fsMutexRelease().
 */
void __libc_back_fsMutexRelease(void);

/**
 * Updates the global unix root stuff.
 * Assumes caller have locked the fs stuff.
 *
 * @param   pszUnixRoot     The new unix root. Fully resolved and existing.
 */
void __libc_back_fsUpdateUnixRoot(const char *pszUnixRoot);

/**
 * Resolves and verifies the user path to a native path.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszUserPath     The user path.
 * @parm    fFlags          Flags controlling the operation of the function.
 *                          See the BACKFS_FLAGS_* defines.
 * @param   pszNativePath   Where to store the native path. This buffer is at
 *                          least PATH_MAX bytes big.
 * @param   pfInUnixTree    Where to store the result-in-unix-tree indicator. Optional.
 */
int __libc_back_fsResolve(const char *pszUserPath, unsigned fFlags, char *pszNativePath, int *pfInUnixTree);

/** __libc_back_fsResolve() flags.
 * @{ */
/** Resolves the path up to but not including the last component. */
#define BACKFS_FLAGS_RESOLVE_PARENT     0x1
/** Resolves and verfies the entire path. */
#define BACKFS_FLAGS_RESOLVE_FULL       0x2
/** Resolves and verfies the entire path but it's ok if the last component doesn't exist. */
#define BACKFS_FLAGS_RESOLVE_FULL_MAYBE 0x3
/** @} */


/**
 * Creates the unix EAs for a new file or directory.
 * @returns 0 on success.
 * @returns Negative errno on failure.
 * @param   hFile           File handle to the newly created fs object. If no
 *                          handle handy, set to -1.
 * @param   pszNativePath   Native path of the newly created fs object. If
 *                          handle is give this will be ignored.
 * @param   mode            The specified file permission mode mask.
 */
int __libc_back_fsSetUnixAttribs(int hFile, const char *pszNativePath, mode_t mode);


/**
 * Creates a symbolic link.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszTarget       The target of the symlink link.
 * @param   pszNativePath   The path to the symbolic link to create.
 */
int __libc_back_fsNativeSymlinkCreate(const char *pszTarget, const char *pszNativePath);

/**
 * Reads the content of a symbolic link.
 *
 * This is weird interface as it will return a truncated result if not
 * enough buffer space. It is also weird in that there is no string
 * terminator.
 *
 * @returns Number of bytes returned in pachBuf.
 * @returns -1 and errno on failure.
 * @param   pszNativePath   The path to the symlink to read.
 * @param   pachBuf         Where to store the symlink value.
 * @param   cchBuf          Size of buffer.
 */
int __libc_back_fsNativeSymlinkRead(const char *pszNativePath, char *pachBuf, size_t cchBuf);

/**
 * Stats a native file.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszNativePath   Path to the file to stat. This path is resolved, no
 *                          processing required.
 * @param   pStat           Where to store the file stats.
 */
int __libc_back_fsNativeFileStat(const char *pszNativePath, struct stat *pStat);


__END_DECLS

#endif
