/* $Id: b_fsFileStatFH.c 1519 2004-09-27 02:15:07Z bird $ */
/** @file
 *
 * LIBC SYS Backend - fstat.
 *
 * Copyright (c) 2003-2004 knut st. osmundsen <bird@innotek.de>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#define INCL_FSMACROS
#include <os2emx.h>
#include "fs.h"
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <emx/io.h>
#include <emx/syscalls.h>
#include <limits.h>
#include "syscalls.h"
#include <InnoTekLIBC/backend.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_BACK_IO
#include <InnoTekLIBC/logstrict.h>



/**
 * Gets the file stats for a file by filehandle.
 *
 * @returns 0 on success.
 * @returns Negative error code (errno.h) on failure. The content
 *          of *pStat is undefined.
 * @param   fh      Handle to file.
 * @param   pStat   Where to store the stats.
 */
int __libc_Back_fsFileStatFH(int fh, struct stat *pStat)
{
    LIBCLOG_ENTER("fh=%d pStat=%p\n", fh, (void *)pStat);

    /*
     * Get filehandle.
     */
    PLIBCFH pFH;

    int rc = __libc_FHEx(fh, &pFH);
    if (!pFH)
        LIBCLOG_RETURN_INT(rc);

    if (!pFH->pOps)
    {
        /*
         * Standard OS/2 file handle.
         *
         * Use query handle type to figure out the file type.
         */
        bzero(pStat, sizeof(*pStat));
        switch (pFH->fFlags & __LIBC_FH_TYPEMASK)
        {
            case F_DEV:     pStat->st_mode = S_IFCHR; break;
            case F_SOCKET:  pStat->st_mode = S_IFSOCK; break;
            /** @todo needs to check if it's a named pipe or not. */
            /* Anonymous pipes are AF_UNIX sockets on (some?) unix systems. */
            case F_PIPE:    pStat->st_mode = S_IFSOCK; break;
            default:
            case F_FILE:    pStat->st_mode = S_IFREG; break;
        }

        FS_VAR();
        FS_SAVE_LOAD();
        if (pStat->st_mode == S_IFREG)
        {
            union
            {
                FILESTATUS3     fsts3;
                FILESTATUS3L    fsts3L;
            } info;
#if OFF_MAX > LONG_MAX
            int     fLarge = 0;
#endif

            /*
             * Get file info.
             */
#if OFF_MAX > LONG_MAX
            if (__libc_gpfnDosOpenL)
            {
                rc = DosQueryFileInfo(fh, FIL_STANDARDL, &info, sizeof(info.fsts3L));
                fLarge = 1;
            }
            else
#endif
                rc = DosQueryFileInfo(fh, FIL_STANDARD, &info, sizeof(info.fsts3));
            if (rc)
            {
                FS_RESTORE();
                rc = -__libc_native2errno(rc);
                LIBCLOG_RETURN_INT(rc);
            }

            /*
             * Format stats struct.
             *      We know the info struct layouts!
             *      Only cbFile, cbFileAlloc and attrFile need be accessed
             *      using the specific structure.
             */
            /* Times: FAT might not return create and access time. */
            pStat->st_mtime = _sys_p2t(info.fsts3.ftimeLastWrite, info.fsts3.fdateLastWrite);
            if (    FTIMEZEROP(info.fsts3.ftimeCreation)
                &&  FDATEZEROP(info.fsts3.fdateCreation))
                pStat->st_ctime = pStat->st_mtime;
            else
                pStat->st_ctime = _sys_p2t(info.fsts3.ftimeCreation, info.fsts3.fdateCreation);
            if (    FTIMEZEROP(info.fsts3.ftimeLastAccess)
                &&  FDATEZEROP(info.fsts3.fdateLastAccess))
                pStat->st_atime = pStat->st_mtime;
            else
                pStat->st_atime = _sys_p2t(info.fsts3.ftimeLastAccess, info.fsts3.fdateLastAccess);

#if OFF_MAX > LONG_MAX
            if (fLarge)
            {
                pStat->st_size = info.fsts3L.cbFile;
                pStat->st_blocks = info.fsts3L.cbFileAlloc / S_BLKSIZE;
                rc = info.fsts3L.attrFile;
            }
            else
#endif
            {
                pStat->st_size = info.fsts3.cbFile;
                pStat->st_blocks = info.fsts3.cbFileAlloc / S_BLKSIZE;
                rc = info.fsts3.attrFile;
            }
            pStat->st_attr = rc;
            if (rc & FILE_READONLY)
                pStat->st_mode |= (S_IREAD >> 6) * 0111;
            else
                pStat->st_mode |= ((S_IREAD|S_IWRITE) >> 6) * 0111;
        }
        else
        {
            if ((pFH->fFlags & O_ACCMODE) == O_RDONLY)
                pStat->st_mode |= (S_IREAD >> 6) * 0111;
            else
                pStat->st_mode |= ((S_IREAD|S_IWRITE) >> 6) * 0111;
        }

        /* default fake stuff */
        /** @todo read EAs containing this info! */
        pStat->st_uid = 0;
        pStat->st_gid = 0;
        pStat->st_ino = _sys_ino++;
        if (_sys_ino == 0)
            _sys_ino = 1;
        pStat->st_dev = 0;
        pStat->st_rdev = 0;
        pStat->st_nlink = 1;
        pStat->st_blksize = 4096*12; /* 48KB */
        FS_RESTORE();
    }
    else
    {
        /*
         * Non-standard handle - fail.
         */
        LIBCLOG_RETURN_INT(-EOPNOTSUPP);
    }

    LIBCLOG_MSG("st_dev:     %#x\n",  pStat->st_dev);
    LIBCLOG_MSG("st_ino:     %#x\n",  pStat->st_ino);
    LIBCLOG_MSG("st_mode:    %#x\n",  pStat->st_mode);
    LIBCLOG_MSG("st_nlink:   %u\n",   pStat->st_nlink);
    LIBCLOG_MSG("st_uid:     %u\n",   pStat->st_uid);
    LIBCLOG_MSG("st_gid:     %u\n",   pStat->st_gid);
    LIBCLOG_MSG("st_rdev:    %#x\n",  pStat->st_rdev);
    LIBCLOG_MSG("st_atime:   %d\n",   pStat->st_atime);
    LIBCLOG_MSG("st_mtime:   %d\n",   pStat->st_mtime);
    LIBCLOG_MSG("st_ctime:   %d\n",   pStat->st_ctime);
    LIBCLOG_MSG("st_size:    %lld\n", pStat->st_size);
    LIBCLOG_MSG("st_blocks:  %lld\n", pStat->st_blocks);
    LIBCLOG_MSG("st_blksize: %u\n",   pStat->st_blksize);
    LIBCLOG_MSG("st_attr:    %ld\n",  pStat->st_attr);
    LIBCLOG_RETURN_INT(0);
}

