/* $Id: thread_internals.c 1519 2004-09-27 02:15:07Z bird $ */
/** @file
 *
 * LIBC - Thread internals.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#undef NDEBUG
#include <386/builtin.h>
#include <assert.h>
#include <string.h>
#include <errno.h>
#include <sys/smutex.h>
#include <emx/umalloc.h>
#include <emx/syscalls.h>
#include <InnoTekLIBC/thread.h>
#include <InnoTekLIBC/backend.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_THREAD
#include <InnoTekLIBC/logstrict.h>


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** Preallocated thread structure. */
static __LIBC_THREAD    gPreAllocThrd;
/** Flags whether or not gPreAllocThrd is used. */
static int              gfPreAllocThrd;

/** Head of the thread termination callback list. */
static __LIBC_PTHREADTERMCBREGREC   gpTermHead;
/** Tail of the thread termination callback list. */
static __LIBC_PTHREADTERMCBREGREC   gpTermTail;
/** List protection semaphore (spinlock sort). */
static _smutex                      gsmtxTerm;



/**
 * Initialize a thread structure.
 *
 * @param   pThrd   Pointer to the thread structure which is to be initialized.
 */
static void threadInit(__LIBC_PTHREAD pThrd)
{
    bzero(pThrd, sizeof(*pThrd));
    pThrd->iRand = 1;
    __libc_Back_threadInit(pThrd);
}


__LIBC_PTHREAD __libc_threadCurrentSlow(void)
{
    LIBCLOG_ENTER("\n");
    if (!*__libc_gpTLS)
    {
        __LIBC_PTHREAD  pThrd;
        /*
         * Setup a temporary thread block on the stack so _hmalloc()
         * can't end up calling us recursivly if something goes wrong
         */
        __LIBC_THREAD   Thrd;
        threadInit(&Thrd);
        *__libc_gpTLS   = &Thrd;

        if (!__lxchg(&gfPreAllocThrd, 1))
            pThrd = &gPreAllocThrd;
        else
        {
            pThrd = _hmalloc(sizeof(__LIBC_THREAD));
            assert(pThrd); /* deep, deep, depp, shit. abort the process in a controlled manner... */
        }
        *pThrd = Thrd;

        *__libc_gpTLS = pThrd;
        LIBCLOG_MSG("Created thread block %p\n", pThrd);
    }

    LIBCLOG_RETURN_P(*__libc_gpTLS);
}


__LIBC_PTHREAD __libc_threadAlloc(void)
{
    LIBCLOG_ENTER("\n");
    /*
     * No need to use the pre allocated here since the current thread will
     * most likely be using that one!
     */
    __LIBC_PTHREAD pThrd = _hmalloc(sizeof(__LIBC_THREAD));
    if (pThrd)
        threadInit(pThrd);
    LIBCLOG_RETURN_P(pThrd);
}


void  __libc_threadFree(__LIBC_PTHREAD pThrd)
{
    LIBCLOG_ENTER("pThrd=%p\n", (void *)pThrd);
    /*
     * Clean up members.
     */
    __libc_Back_threadCleanup(pThrd);

    /*
     * Release storage.
     */
    if (pThrd != &gPreAllocThrd)
        free(pThrd);
    else
        __lxchg(&gfPreAllocThrd, 0);
    LIBCLOG_RETURN_VOID();
}


int     __libc_ThreadRegisterTermCallback(__LIBC_PTHREADTERMCBREGREC pRegRec)
{
    LIBCLOG_ENTER("pRegRec=%p {%p, %u, %p}\n", (void *)pRegRec,
                  pRegRec ? (void *)pRegRec->pNext : NULL,
                  pRegRec ? pRegRec->fFlags : 0,
                  pRegRec ? (void *)pRegRec->pfnCallback : NULL);

    /*
     * Validate input.
     */
    if (    pRegRec->pNext
        ||  !pRegRec->pfnCallback
        ||  pRegRec->fFlags)
    {
        LIBC_ASSERTM(!pRegRec->pNext, "pNext must be NULL not %p\n", pRegRec->pNext);
        LIBC_ASSERTM(pRegRec->pfnCallback, "pfnCallback not be NULL\n");
        LIBC_ASSERTM(!pRegRec->fFlags, "fFlags must be ZERO not %u\n", pRegRec->fFlags);
        errno = EINVAL;
        LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Insert into the LIFO.
     */
    _smutex_request(&gsmtxTerm);
    if (    !pRegRec->pNext
        &&  pRegRec != gpTermTail)
    {
        pRegRec->pNext = gpTermTail;
        gpTermTail = pRegRec;
        if (!gpTermHead)
            gpTermHead = pRegRec;
        _smutex_release(&gsmtxTerm);
        LIBCLOG_RETURN_INT(0);
    }
    else
    {
        _smutex_release(&gsmtxTerm);
        LIBC_ASSERTM_FAILED("Double registration of %p\n", pRegRec);
        LIBCLOG_RETURN_INT(-1);
    }
}


void    __libc_threadTermination(unsigned fFlags)
{
    LIBCLOG_ENTER("fFlags=%u\n", fFlags);
    __LIBC_PTHREADTERMCBREGREC  pCur;
    __LIBC_PTHREADTERMCBREGREC  pTail;

    /*
     * We'll need to do this in a safe manner.
     *
     * ASSUME no removal of records.
     * Thus we just pick the head and tail record and walk them.
     */
    _smutex_request(&gsmtxTerm);
    pTail = gpTermTail;
    pCur = gpTermHead;
    _smutex_release(&gsmtxTerm);

    while (pCur)
    {
        /* call it */
        LIBCLOG_MSG("calling %p with %p\n", pCur->pfnCallback, pCur);
        pCur->pfnCallback(pCur, fFlags);

        /* next */
        if (pCur == pTail)
            break;
        pCur = pCur->pNext;
    }

    LIBCLOG_RETURN_VOID();
}
