/* sys/fcntl.c (emx+gcc) -- Copyright (c) 1992-1996 by Eberhard Mattes
                         -- Copyright (c) 2003 by Knut St. Osmunden */

#include "libc-alias.h"
#include <fcntl.h>
#include <errno.h>
#include <limits.h>
#include <io.h>
#include <string.h>
#define INCL_ERRORS
#define INCL_FSMACROS
#include <os2emx.h>
#include <386/builtin.h>
#include <emx/io.h>
#include <emx/syscalls.h>
#include "syscalls.h"
#define __LIBC_LOG_GROUP    __LIBC_LOG_GRP_IO
#include <InnoTekLIBC/logstrict.h>


/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int __fcntl_getfd(__LIBC_PFH pFH, int hFile);
static int __fcntl_setfd(__LIBC_PFH pFH, int hFile, int arg);
static int __fcntl_locking(int hFile, int iRequest, struct flock *pFlock);


int __fcntl(int hFile, int iRequest, int arg)
{
    LIBCLOG_ENTER("hFile=%d iRequest=%#x arg=%#x\n", hFile, iRequest, arg);
    int         rc;
    __LIBC_PFH  pFH;

    /*
     * Get the file hFile data.
     */
    pFH = __libc_FH(hFile);
    if (!pFH)
    {
        errno = EBADF;
        LIBCLOG_RETURN_INT(-1);
    }

    if (!pFH->pOps)
    {
        /*
         * Standard OS/2 File hFile.
         */
        switch (iRequest)
        {
            /*
             * Get file status flags and access modes.
             */
            case F_GETFL:
            {
                unsigned fFlags = pFH->fFlags & (O_ACCMODE | O_APPEND | O_NONBLOCK | O_SYNC /*| O_*SYNC*/);
                LIBCLOG_RETURN_INT(fFlags);
            }

            /*
             * Set file status flags.
             */
            case F_SETFL:
            {
                /** @todo implement this properly. See FCNTLFLAGS. */
                LIBC_ASSERTM_FAILED("F_SETFL isn't implemented but returns success. arg=%#x\n", arg);
                LIBCLOG_RETURN_INT(0);
            }

            /*
             * Get file descriptor flags.
             */
            case F_GETFD:
                rc = __fcntl_getfd(pFH, hFile);
                LIBCLOG_RETURN_INT(rc);

            /*
             * Set file descriptor flags.
             */
            case F_SETFD:
                rc = __fcntl_setfd(pFH, hFile, arg);
                LIBCLOG_RETURN_INT(rc);

            /*
             * File locking.
             */
            case F_GETLK:   /* get record locking information */
            case F_SETLK:   /* set record locking information */
            case F_SETLKW:  /* F_SETLK; wait if blocked */
            {
                int rc = __fcntl_locking(hFile, iRequest, (struct flock*)arg);
                LIBCLOG_RETURN_INT(rc);
            }

            default:
                LIBC_ASSERTM_FAILED("Invalid iRequest %#x\n", iRequest);
                errno = EINVAL;
                LIBCLOG_RETURN_INT(-1);
        }
    }
    else
    {
        /*
         * Non-standard hFile - call registered method.
         */
        int rcRet = 0;
        rc = pFH->pOps->pfnFileControl(pFH, hFile, iRequest, arg, &rcRet);
        if (rc)
        {
            if (rc > 0)
                _sys_set_errno(rc);
            else
                errno = -rc;
            LIBCLOG_RETURN_INT(-1);
        }

        /*
         * Post process to keep the OS/2 fake hFile up to date (on success).
         */
        switch (iRequest)
        {
            case F_SETFD:
                rc = __fcntl_setfd(pFH, hFile, arg);
                if (rc == -1)
                    LIBCLOG_RETURN_INT(-1);
                break;
        }
        LIBCLOG_RETURN_INT(rcRet);
    }
}



/**
 * F_GETFD operation on standard OS/2 hFile.
 * Gets file descriptor flags, which at the moment is limited to FD_CLOEXEC.
 *
 * @returns 0 on success.
 * @returns -1 an errno on failure.
 * @param   pFH     File handler structure.
 * @param   hFile  File hFile.
 */
static int __fcntl_getfd(__LIBC_PFH pFH, int hFile)
{
    LIBCLOG_ENTER("pFH=%p hFile=%d\n", (void *)pFH, hFile);
    int     rc;
    ULONG   fulState;
    FS_VAR();

    FS_SAVE_LOAD();
    rc = DosQueryFHState(hFile, &fulState);
    FS_RESTORE();
    if (!rc)
    {
        unsigned fFlags = pFH->fFlags;
        /* flags out of sync? */
        if (    ( (fulState & OPEN_FLAGS_NOINHERIT) != 0 )
            !=  (   (fFlags & (O_NOINHERIT | (FD_CLOEXEC << __LIBC_FH_FDFLAGS_SHIFT)))
                 == (O_NOINHERIT | (FD_CLOEXEC << __LIBC_FH_FDFLAGS_SHIFT)) ) )
        {
            LIBC_ASSERTM_FAILED("Inherit flags are out of sync for file hFile %d (%#x)! fulState=%08lx fFlags=%08x\n",
                                hFile, hFile, fulState, fFlags);
            if (fulState & OPEN_FLAGS_NOINHERIT)
                fFlags |= O_NOINHERIT | FD_CLOEXEC;
            else
                fFlags &= ~(O_NOINHERIT | FD_CLOEXEC);
            __atomic_xchg(&pFH->fFlags, fFlags);
        }

        fFlags >>= __LIBC_FH_FDFLAGS_SHIFT;
        LIBCLOG_RETURN_INT(fFlags);
    }

    /* failure. */
    _sys_set_errno(rc);
    LIBCLOG_RETURN_INT(-1);
}


/**
 * F_SETFD operation on standard OS/2 hFile.
 * Sets file descriptor flags, which at the moment is limited to FD_CLOEXEC.
 *
 * @returns 0 on success.
 * @returns -1 an errno on failure.
 * @param   pFH     File handler structure.
 * @param   hFile  File hFile.
 * @param   arg     New file descriptor flags.
 */
static int __fcntl_setfd(__LIBC_PFH pFH, int hFile, int arg)
{
    LIBCLOG_ENTER("pFH=%p hFile=%d arg=%#x\n", (void *)pFH, hFile, arg);
    int     rc;
    ULONG   fulState;
    FS_VAR();

    FS_SAVE_LOAD();
    rc = DosQueryFHState(hFile, &fulState);
    if (!rc)
    {
        ULONG fulNewState;
        LIBC_ASSERTM(     ( (fulState & OPEN_FLAGS_NOINHERIT) != 0 )
                      ==  (   (pFH->fFlags & (O_NOINHERIT | (FD_CLOEXEC << __LIBC_FH_FDFLAGS_SHIFT)))
                           == (O_NOINHERIT | (FD_CLOEXEC << __LIBC_FH_FDFLAGS_SHIFT)) ),
                     "Inherit flags are out of sync for file hFile %d (%#x)! fulState=%08lx fFlags=%08x\n",
                     hFile, hFile, fulState, pFH->fFlags);
        if (arg & FD_CLOEXEC)
            fulNewState = fulState | OPEN_FLAGS_NOINHERIT;
        else
            fulNewState = fulState & ~OPEN_FLAGS_NOINHERIT;
        if (fulNewState != fulState)
        {
            fulNewState &= 0x7f88; /* Mask the flags accepted the API. */
            rc = DosSetFHState(hFile, fulNewState);
        }
    }
    FS_RESTORE();

    if (!rc)
    {
        unsigned fFlags = pFH->fFlags;
        fFlags = (fFlags & ~__LIBC_FH_FDFLAGS_MASK) | (arg << __LIBC_FH_FDFLAGS_SHIFT);
        if (arg & FD_CLOEXEC)
            fFlags |= O_NOINHERIT;
        else
            fFlags &= ~O_NOINHERIT;
        __atomic_xchg(&pFH->fFlags, fFlags);
        LIBCLOG_RETURN_INT(0);
    }

    /* error! */
    _sys_set_errno(rc);
    LIBCLOG_RETURN_INT(-1);
}


/**
 * Handle locking requests.
 * @returns
 * @param   hFile       File hFile.
 * @param   iRequest     Lock iRequest.
 * @param   pFlock      Pointer to flock structure.
 */
static int __fcntl_locking(int hFile, int iRequest, struct flock *pFlock)
{
    APIRET        rc;
    union
    {
        FILESTATUS3     fsts3;
        FILESTATUS3L    fsts3L;
    } info;
#if OFF_MAX > LONG_MAX
    int     fLarge = 0;
#endif
    FS_VAR();

    /* check input */
    /** @todo: Implement F_GETLK */
    if (!pFlock || iRequest == F_GETLK)
    {
        errno = EINVAL;
        return -1;
    }

    /* check hFile & get filesize. */
    FS_SAVE_LOAD();
#if OFF_MAX > LONG_MAX
    if (__pfnDosOpenL)
    {
        rc = DosQueryFileInfo(hFile, FIL_STANDARDL, &info, sizeof(info.fsts3L));
        fLarge = 1;
    }
    else
#endif
        rc = DosQueryFileInfo(hFile, FIL_STANDARD, &info, sizeof(info.fsts3));
    FS_RESTORE();
    if (!rc)
    {
        ULONG       fAccess;
        int         fLock;
        ULONG       ulTimeout;
        off_t       cbFile;
        off_t       offStart;
        off_t       cbRange;
#if OFF_MAX > LONG_MAX
        if (fLarge)
            cbFile = info.fsts3L.cbFile;
        else
#endif
            cbFile = info.fsts3.cbFile;

        /* range */
        cbRange = pFlock->l_len ? pFlock->l_len : OFF_MAX;

        /* offset */
        switch (pFlock->l_whence)
        {
            case SEEK_SET:  offStart = pFlock->l_start; break;
            case SEEK_CUR:  offStart = tell(hFile) + pFlock->l_start; break;
            case SEEK_END:  offStart = cbFile - pFlock->l_start; break;
            default:
                errno = EINVAL;
                return -1;
        }
        if (    offStart < 0
            ||  cbRange + offStart < 0)
        {
            errno = EINVAL;
            return -1;
        }

        /* flags and order */
        fAccess = 0; /* exclusive */
        switch (pFlock->l_type)
        {
            case F_UNLCK:
                fLock = 0;
                break;

            case F_RDLCK:
                fAccess = 1; /* shared */
            case F_WRLCK:
                fLock = 1;
                break;

            default:
                errno = EINVAL;
                return -1;
        }

        /* timeout */
        if (iRequest == F_SETLKW)
            ulTimeout = SEM_INDEFINITE_WAIT;
        else
            ulTimeout = SEM_IMMEDIATE_RETURN;

        /* Do work. */
#if OFF_MAX > LONG_MAX
        rc = ERROR_INVALID_PARAMETER;
        if (__pfnDosSetFileLocksL)
        {
            FILELOCKL   aflock[2];
            bzero(&aflock[(fLock + 1) & 1], sizeof(aflock[0]));
            aflock[fLock].lOffset = offStart;
            aflock[fLock].lRange  = cbRange;
            FS_SAVE_LOAD();
            rc = __pfnDosSetFileLocksL(hFile, &aflock[0], &aflock[1], ulTimeout, fAccess);
            FS_RESTORE();
        }
        /* There is/was a bug in the large API which make it fail on non JFS
         * disks with ERROR_INVALID_PARAMETER. We need to work around this. */
        if (rc == ERROR_INVALID_PARAMETER)
#endif
        {
            FILELOCK    aflock[2];
#if OFF_MAX > LONG_MAX
            if (    offStart > LONG_MAX
                ||  (   cbRange != OFF_MAX
                     && (   cbRange > LONG_MAX
                         || offStart + cbRange > LONG_MAX)
                    )
               )
            {
                errno = EOVERFLOW;
                return -1;
            }
#endif
            bzero(&aflock[(fLock + 1) & 1], sizeof(aflock[0]));
            aflock[fLock].lOffset = offStart;
            aflock[fLock].lRange  = cbRange;
            FS_SAVE_LOAD();
            rc = DosSetFileLocks(hFile, &aflock[0], &aflock[1], ulTimeout, fAccess);
            FS_RESTORE();
        }
    }

    /* done */
    if (!rc)
        return 0;
    _sys_set_errno (rc);
    return -1;
}

