/* $Id: backend.h 1517 2004-09-19 15:53:31Z bird $ */
/** @file
 *
 * LIBC - Backend header.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __InnoTekLIBC_backend_h__
#define __InnoTekLIBC_backend_h__

#include <sys/cdefs.h>
#include <sys/types.h>

__BEGIN_DECLS

#ifndef __LIBC_THREAD_DECLARED
#define __LIBC_THREAD_DECLARED
typedef struct __libc_thread *__LIBC_PTHREAD, **__LIBC_PPTHREAD;
#endif
struct statfs;


/** @defgroup __libc_Back_thread   LIBC Backend - Threads
 * @{ */

/**
 * Initiatlize a new thread structure.
 *
 * @param   pThrd   Pointer to the thread structure.
 */
void __libc_Back_threadInit(__LIBC_PTHREAD pThrd);

/**
 * Called before the thread structure is freed so the backend
 * can cleanup its members.
 *
 * @param   pThrd   Pointer to the thread in  question.
 */
void __libc_Back_threadCleanup(__LIBC_PTHREAD pThrd);

/**
 * Called in the context of the newly started thread to register
 * exception handler and to do other init stuff.
 *
 * @param   pExpRegRec  Exception handler registration record on the stack.
 *                      To be used for any exception handler registration.
 */
void __libc_Back_threadStartup(void *pExpRegRec);

/**
 * Called in the context of the thread which is to be terminated to
 * unregister exception handler and to do other final term stuff.
 *
 * @param   pExpRegRec  Exception handler registration record on the stack.
 *                      To be used for any exception handler registration.
 * @remark  This is called after __libc_Back_threadCleanup().
 * @remark  It is not called by thread which calls _endthread(), nor for the
 *          main thread.
 */
void __libc_Back_threadEnd(void *pExpRegRec);

/** @} */


/** @defgroup __libc_Back_fh   LIBC Backend - File Handles.
 * @{ */

/**
 * Try resolve a filehandle to a path.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   fh          The file handle.
 * @param   pszPath     Where to store the path.
 * @param   cchPath     The size of he buffer pointed to by pszPath.
 */
int __libc_Back_fhToPath(int fh, char *pszPath, size_t cchPath);

/** @} */


/** @defgroup __libc_Back_fs   LIBC Backend - File System
 * @{ */

/**
 * Get the statistics for the filesystem which pszPath is located on.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszPath     The path to somewhere in the filesystem.
 * @param   pStatFs     Where to store the obtained information.
 */
int __libc_Back_fsStat(const char *pszPath, struct statfs *pStatFs);

/**
 * Get file system statistics
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   fh          The filehandle of any file within the mounted file system.
 * @param   pStatFs     Where to store the statistics.
 */
int __libc_Back_fsStatFH(int fh, struct statfs *pStatFs);

/**
 * Get the statistics for all the mounted filesystems.
 *
 * @returns Number of returned statfs structs on success.
 * @returns Number of mounted filesystems on success if paStatFS is NULL
 * @returns -1 and errno on failure.
 * @param   paStatFs    Where to to store the statistics.
 * @param   cStatFS     Number of structures the array pointed to by paStatFs can hold.
 * @param   fFlags      Flags, currently ignored.
 */
int __libc_Back_fsStats(struct statfs *paStatFs, unsigned cStatFs, unsigned fFlags);

/**
 * Schedules all file system buffers for writing.
 *
 * See sync() for OS/2 limitations.
 */
void __libc_Back_fsSync(void);

/**
 * Sets or change the unixroot of the current process.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pszNewRoot  The new root.
 */
int __libc_Back_fsChRoot(const char *pszNewRoot);

/** @} */



/** @defgroup __libc_Back_misc   LIBC Backend - Miscellaneous
 * @{ */

/**
 * Gets the system load averages.
 * The load is the average values of ready and running threads(/processes)
 * over the last 1, 5 and 15 minuttes.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pardAvgs    Where to store the samples.
 * @param   cAvgs       Number of samples to get. Max is 3.
 * @remark  See OS/2 limitations in getloadavg().
 */
int __libc_Back_miscLoadAvg(double *pardAvgs, unsigned cAvgs);

/** @} */

__END_DECLS

#endif
