/** $Id: sharedpm.h 1461 2004-09-06 04:22:46Z bird $ */
/** @file
 *
 * LIBC Shared Process Management.
 *
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __InnoTekLIBC_sharedpm_h__
#define __InnoTekLIBC_sharedpm_h__

#include <sys/cdefs.h>
#include <sys/types.h>

__BEGIN_DECLS

/** The name of the shared semaphore protecting the memory. */
#define SPM_MUTEX_NAME          "\\SEM32\\INNOTEKLIBC.V01"

/** The name of the shared memory. */
#define SPM_MEMORY_NAME         "\\SHAREMEM\\INNOTEKLIBC.V01"

/** The timeout for accessing the shared mem semaphore. */
#define SPM_MUTEX_TIMEOUT       (30*1000)

/** Allocate 512KB if OBJ_ANY isn't supported.
 * If it is supported we allocated the double amount. */
#define SPM_MEMORY_SIZE         (512*1024)

/** The defined max size of a process.
 * A good bit is reserved for the future here.
 *
 * Theoretically speaking, OS/2 cannot entertain more than a 1000 processes.
 * So, even if we reserve a good bit here it's not gonna cause any trouble.
 */
#define SPM_PROCESS_SIZE        (256)

/**
 * The SPM version.
 */
#define SPM_VERSION             0x00010000

/**
 * Process state.
 */
typedef enum __LIBC_SPMPROCSTAT
{
    /** The process is free. */
    __LIBC_PROCSTATE_FREE = 0,
    /** The process is waiting for a chile to be created and snatch all the
     * inherited goodies from it.
     * If not caught before DosExecPgm/DosStartSession returns
     * it will be freed by the disapointed parent.
     */
    __LIBC_PROCSTATE_EMBRYO = 1,
    /** The process is running. */
    __LIBC_PROCSTATE_ALIVE = 2,
    /** The process is dead but other guys are still using it's data. */
    __LIBC_PROCSTATE_ZOMBIE = 3,
    /** The maximum state number (exclusive). */
    __LIBC_PROCSTATE_MAX = 10,
    /** Hack to ensure that this will be a 32-bit int. */
    __LIBC_PROCSTATE_MAKE_INT32 = 0x7fffffff
} __LIBC_SPMPROCSTAT;

/**
 * Process termination reason.
 */
typedef enum __LIBC_EXIT_REASON
{
    /** No reason. */
    __LIBC_EXIT_REASON_NONE = 0,
    /** Normal exit. */
    __LIBC_EXIT_REASON_EXIT = 1,
    /** OS/2 hard error. */
    __LIBC_EXIT_REASON_HARDERROR = 2,
    /** Trap (i.e. 16-bit?). */
    __LIBC_EXIT_REASON_TRAP = 3,
    /** Kill process. */
    __LIBC_EXIT_REASON_KILL = 4,
    /** Exception. */
    __LIBC_EXIT_REASON_XCPT = 5,
    /** Signal base. */
    __LIBC_EXIT_REASON_SIGNAL_BASE = 32,
    /** Signal max. */
    __LIBC_EXIT_REASON_SIGNAL_MAX = 256,
    /** Hack to ensure that this will be a 32-bit int. */
    __LIBC_EXIT_REASON_MAKE_INT32 = 0x7fffffff
} __LIBC_EXIT_REASON;

/** Inheritance FH Bundle Type.
 * @{ */
/** Termination bundle. */
#define __LIBC_SPM_INH_FHB_TYPE_END         (0)
/** Standard bundle. */
#define __LIBC_SPM_INH_FHB_TYPE_STANDARD    (0x10 | sizeof(unsigned))
/** Socket bundle, using the BSD 4.4 backend. */
#define __LIBC_SPM_INH_FHB_TYPE_SOCKET_44   (0x20 | (sizeof(unsigned) + sizeof(unsigned short)))
/** Socket bundle, using the BSD 4.3 backend. */
#define __LIBC_SPM_INH_FHB_TYPE_SOCKET_43   (0x30 | (sizeof(unsigned) + sizeof(unsigned short)))
/** Get the per file handle size from the bundle type. */
#define __LIBC_SPM_INH_FHB_SIZE(type)   ((type) & 0x0f)
/** @} */

/**
 * SPM Inheritance File Handle Bundle Header.
 */
#pragma pack(1)
typedef struct __libc_SPMInhFHBHdr
{
    /** Bundle type. */
    unsigned char   uchType;
    /** Cound of handles in this bundle. */
    unsigned char   cHandles;
    /** Start handle number. */
    unsigned short  fhStart;
} __LIBC_SPMINHFHBHDR;
#pragma pack()
/** Pointer to SPM filehandle bundle header. */
typedef __LIBC_SPMINHFHBHDR *__LIBC_PSPMINHFHBHDR;

/**
 * SPM standard filehandle inherit bundle
 * This is used for OS/2 filehandles which only needs flags
 * transfered.
 */
#pragma pack(1)
typedef struct __libc_SPMInhFH
{
    /** The common bundle header. */
    __LIBC_SPMINHFHBHDR Hdr;
    /** Array of flags of Hdr.cHandles entries. */
    unsigned        afFlags[1];
} __LIBC_SPMINHFHBSTD;
#pragma pack()
/** Pointer to SPM standard filehandle inherit bundle. */
typedef __LIBC_SPMINHFHBSTD *__LIBC_PSPMINHFHBSTD;

/**
 * SPM socket filehandle inherit bundle.
 *
 * This is used for both BSD 4.3 and 4.4 sockets. The data needed
 * here is the flags and the socket number.
 */
#pragma pack(1)
typedef struct __libc_SPMInhFHSocket
{
    /** The common bundle header. */
    __LIBC_SPMINHFHBHDR Hdr;
    /** Array of flags of Hdr.cHandles entries. */
    struct
    {
        /** The file handle flags. */
        unsigned        fFlags;
        /** The socket number. */
        unsigned short  usSocket;
    } aHandles[1];
} __LIBC_SPMINHFHBSOCK;
#pragma pack()
/** Pointer to SPM socket filehandle inherit bundle. */
typedef __LIBC_SPMINHFHBSOCK *__LIBC_PSPMINHFHBSOCK;


/**
 * Inherit structure.
 *
 * All the data it contains is allocated in one block heap block!
 */
typedef struct __libc_SPMInherit
{
    /** Size of the inherit structure. */
    size_t                  cb;
    /** Pointer to the filehandle part.
     * This is a succession of bundles terminating with a _END one. */
    __LIBC_PSPMINHFHBHDR    pFHBundles;

    /** @todo Add signals and the other properties which are inherited. */
} __LIBC_SPMINHERIT;
/** Pointer to inherit data. */
typedef __LIBC_SPMINHERIT *__LIBC_PSPMINHERIT;


/**
 * Per Process Data.
 */
typedef struct __libc_SPMProcess
{
    /** Pointer to the next process in the list.
     * Every process is in one or another list depending on their state. */
    struct __libc_SPMProcess   *pNext;
    /** Pointer to the previous process in the list. */
    struct __libc_SPMProcess   *pPrev;

    /** Version of the SPM which created this process. */
    unsigned                    uVersion;
    /** Number of references to this process.
     * A process is not actually free till the reference count
     * reaches 0. */
    unsigned                    cReferences;
    /** Process id. */
    pid_t                       pid;
    /** Process id of parent. */
    pid_t                       pidParent;
    /** State of this process. */
    __LIBC_SPMPROCSTAT          enmState;
    /** The SPM open count of this process. */
    unsigned                    cSPMOpens;
    /** Exit code. */
    int                         iExitCode;
    /** Reason code. */
    __LIBC_EXIT_REASON          enmDeathReason;
    /** Pointer to fork module list head. */
    void * volatile             pvModuleHead;
    /** Pointer to fork module list tail pointer. */
    void * volatile * volatile  ppvModuleTail;
    /** Pointer to fork handle which a child should use when spawned by fork(). */
    void                       *pvForkHandle;
    /** Creation timestamp. */
    unsigned                    uTimestamp;

    /** Reserved pool pointer field with default value 0. */
    unsigned                    aReserved[40 - 14];

    /** Number of possible pointers to shared memory starting at pvInherit.
     * This means we can extend the structure with pointers for as long as we
     * want and still have older LIBC versions cleanup the mess. */
    unsigned                    cPoolPointers;
    /** Pointer to data inherited from the parent process. */
    __LIBC_PSPMINHERIT volatile pInherit;
    /** Pointer to data inherited from the parent process when it's locked.
     * When locked pInherit is NULL and this member points to the data instead.
     * This prevents spmAlloc() from reclaiming it while it's in use. */
    __LIBC_PSPMINHERIT volatile pInheritLocked;
} __LIBC_SPMPROCESS, *__LIBC_PSPMPROCESS;


/**
 * Pool chunk.
 */
typedef struct __libc_SPMPoolChunk
{
    /** Next block in the list of all blocks.. */
    struct __libc_SPMPoolChunk *pPrev;
    /** Previous block in the list of all blocks.. */
    struct __libc_SPMPoolChunk *pNext;
} __LIBC_SPMPOOLCHUNK, *__LIBC_PSPMPOOLCHUNK;

/**
 * Free pool chunk.
 */
typedef struct __libc_SPMPoolChunkFree
{
    /** Main list. */
    __LIBC_SPMPOOLCHUNK             core;
    /** Next block in list of free nodes. */
    struct __libc_SPMPoolChunkFree *pPrev;
    /** Previous block in list of free nodes. */
    struct __libc_SPMPoolChunkFree *pNext;
    /** Size of this block. */
    size_t                          cb;
} __LIBC_SPMPOOLCHUNKFREE, *__LIBC_PSPMPOOLCHUNKFREE;

/**
 * Pool chunk alignment.
 */
#define SPM_POOL_ALIGN(size)    (((size) + 7) & ~7)

/**
 * This structure contains the global TCP/IP socket data.
 */
typedef struct __libc_SPMTcpIp
{
    /** Size of the structure. */
    size_t                      cb;
    /** The number of elements in the acRefs array. */
    unsigned                    cSockets;
    /** Array containing the references counters for the sockets in the system. */
    uint16_t                    acRefs[32768];
} __LIBC_SPMTCPIP, *__LIBC_PSPMTCPIP;


/**
 * This is the header of the LIBC shared memory.
 */
typedef struct __libc_SPMHeader
{
    /** SPM version. (the one which initialized the memory)  */
    unsigned                    uVersion;

    /** Size of the shared memory. */
    size_t                      cb;
    /** Free memory. */
    size_t                      cbFree;
    /** Start chunk of shared memory pool. */
    __LIBC_PSPMPOOLCHUNK        pPoolHead;
    /** End chunk of shared memory pool. */
    __LIBC_PSPMPOOLCHUNK        pPoolTail;
    /** Start free chunk in the shared memory pool. */
    __LIBC_PSPMPOOLCHUNKFREE    pPoolFreeHead;
    /** End free chunk in the shared memory pool. */
    __LIBC_PSPMPOOLCHUNKFREE    pPoolFreeTail;

    /** The size of one process block in this version of the shared memory. */
    size_t                      cbProcess;
    /** Array index by process state giving the head pointers
     * to the processes in that state. */
    __LIBC_PSPMPROCESS volatile apHeads[__LIBC_PROCSTATE_MAX];

    /** Pointer to the tcpip globals. */
    __LIBC_PSPMTCPIP            pTcpip;
    /** Creator pid. */
    pid_t                       pidCreate;
    /** Creation timestamp. */
    __extension__ union
    {
#ifdef INCL_DOSDATETIME
        DATETIME                dtCreate;
#endif
        char                    ach[16];
    };
    /* The rest of the block, up to cbProcess, is undefined in this version.
     * Future versions of LIBC may use this area assuming it's initalized with zeros.
     */
} __LIBC_SPMHEADER, *__LIBC_PSPMHEADER;


/**
 * SPM Exception handler registration record.
 */
typedef struct __LIBC_SPMXCPTREGREC
{
    __extension__ union
    {
#ifdef INCL_DOSEXCEPTIONS
        EXCEPTIONREGISTRATIONRECORD Core;
#endif
        void *apv[2];
    };
    unsigned                        uFutureStuff0;
    unsigned                        uFutureStuff1;
} __LIBC_SPMXCPTREGREC;
/** Pointer to SPM exception handler registration record. */
typedef __LIBC_SPMXCPTREGREC *__LIBC_PSPMXCPTREGREC;




/**
 * Gets the current process.
 *
 * @returns Pointer to the current process.
 * @returns NULL and errno on failure.
 * @remark  For this too __libc_spmRelease() must be called when done.
 */
__LIBC_PSPMPROCESS __libc_spmSelf(void);

/**
 * Gets the inherit data associated with the current process.
 * This call prevents it from being release by underrun handling.
 *
 * @returns Pointer to inherit data.
 *          The caller must call __libc_spmInheritRelease() when done.
 * @returns NULL and errno if no inherit data.
 */
__LIBC_PSPMINHERIT  __libc_spmInheritRequest(void);

/**
 * Releases the inherit data locked by the __libc_spmInheritRequest() call.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 */
int  __libc_spmInheritRelease(void);

/**
 * Frees the inherit data of this process.
 * This is called when the executable is initialized.
 */
void  __libc_spmInheritFree(void);

/**
 * Create an embryo related to the current process.
 *
 * @returns pointer to the embryo process.
 *          The allocated process must be released by the caller.
 * @returns NULL and errno on failure.
 * @param   pidParent   The parent pid (i.e. this process).
 */
__LIBC_PSPMPROCESS __libc_spmCreateEmbryo(pid_t pidParent);

/**
 * Searches for a process given by pid.
 *
 * @returns Pointer to the desired process on success.
 * @returns NULL and errno on failure.
 * @param   pid         Process id to search for.
 * @param   enmState 	The state of the process.
 * @remark  Call __libc_spmRelease() to release the result.
 */
__LIBC_PSPMPROCESS __libc_spmQueryProcess(pid_t pid);

/**
 * Searches for a process with a given pid and state.
 *
 * @returns Pointer to the desired process on success.
 * @returns NULL and errno on failure.
 * @param   pid         Process id to search for.
 * @param   enmState 	The state of the process.
 * @remark  Call __libc_spmRelease() to release the result.
 */
__LIBC_PSPMPROCESS __libc_spmQueryProcessInState(pid_t pid, __LIBC_SPMPROCSTAT enmState);

/**
 * Release reference to the given process.
 *
 * @returns 0 on success.
 * @returns -1 on failure.
 * @param   pProcess    Pointer to process to release.
 */
int __libc_spmRelease(__LIBC_PSPMPROCESS pProcess);

/**
 * Marks the current process (if we have it around) as zombie
 * or dead freeing all resources associated with it.
 *
 * @param   uReason     The OS/2 exit list type reason code.
 *                      This is only used if the current code is NONE.
 * @param   iExitCode   The unix exit code for this process.
 *                      This is only if the current code is 0.
 * @remark this might not be a sufficient interface for process termination but we'll see.
 */
void __libc_spmTerm(__LIBC_EXIT_REASON enmDeathReason, int iExitCode);

/**
 * Locks the LIBC shared memory for short exclusive access.
 * The call must call __libc_spmUnlock() as fast as possible and make
 * no api calls until that is done.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pRegRec     Pointer to the exception handler registration record.
 * @remark  Don't even think of calling this if you're not LIBC!
 */
int __libc_spmLock(__LIBC_PSPMXCPTREGREC pRegRec);

/**
 * Unlock the LIBC shared memory after call to __libc_spmLock().
 *
 * @returns 0 on success.
 * @returns -1 on and errno failure.
 * @param   pRegRec     Pointer to the exception handler registration record.
 * @remark  Don't even think of calling this if you're not LIBC!
 */
int __libc_spmUnlock(__LIBC_PSPMXCPTREGREC pRegRec);

/**
 * Allocate memory from the LIBC shared memory.
 *
 * @returns Pointer to allocated memory on success.
 * @returns NULL on failure.
 * @param   cbSize 	Size of memory to allocate.
 * @remark  Don't think of calling this if you're not LIBC!
 */
void * __libc_spmAlloc(size_t cbSize);

/**
 * Free memory allocated by __libc_SpmAlloc().
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   pv      Pointer to memory block returned by __libc_SpmAlloc().
 *                  NULL is allowed.
 * @remark  Don't think of calling this if you're not LIBC!
 */
int __libc_spmFree(void *pv);


/**
 * Register TCPIP termination handler.
 *
 * This is a manual way of by passing a.out's broken weak symbols.
 * @param   pfnTerm     Pointer to the termination function.
 */
void    __libc_spmSocketRegTerm(void (*pfnTerm)(void));


/**
 * Adds the first reference to a new socket.
 *
 * @returns 0 on success.
 * @returns -1 and errno on failure.
 * @param   iSocket     The new socket.
 */
int     __libc_spmSocketNew(int iSocket);

/**
 * References a socket.
 *
 * @returns The new reference count.
 * @returns -1 and errno on failure.
 * @param   iSocket     socket to reference.
 */
int     __libc_spmSocketRef(int iSocket);

/**
 * Dereferences a socket.
 *
 * @returns The new reference count.
 *          The caller must close the socket if 0 is returned.
 * @returns -1 and errno on failure.
 * @param   iSocket     Socket to dereference.
 */
int     __libc_spmSocketDeref(int iSocket);

/**
 * Checks the SPM memory for trouble.
 *
 * @returns 0 on perfect state.
 * @returns -1 and errno on mutex failure.
 * @returns Number of failures if SPM is broken.
 * @param   fBreakpoint Raise breakpoint exception if a problem is encountered.
 * @param   fVerbose    Log everything.
 */
int     __libc_SpmCheck(int fBreakpoint, int fVerbose);

__END_DECLS

#endif /* __InnoTekLIBC_sharedpm_h__ */
