/* $Id: tls.c 1290 2004-03-15 01:54:50Z bird $ */
/** @file
 *
 * InnoTek LIBC - Thread Local Storage Implementation.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <sys/builtin.h>
#include <errno.h>
#include <sys/limits.h>
#include <InnoTekLIBC/thread.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_THREAD
#include <InnoTekLIBC/logstrict.h>


#if ULONG_MAX == UINT_MAX
    /* 32 bits */
    #define ENTRY_BITS      32
    #define ENTRY_BYTES     4
    #define INDEX_SHIFT     5
    #define BIT_MASK        0x0000001f
#else
    /* 64 bits */
    #define ENTRY_BITS      64
    #define ENTRY_BYTES     8
    #define INDEX_SHIFT     6
    #define BIT_MASK        0x000000000000003f
#endif


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** TLS allocation bitmap map.
 * Set means allocated, clear means free. Updated atomically. */
static unsigned auBitmap[(__LIBC_TLS_MAX + ENTRY_BITS - 1) / ENTRY_BITS];

/** Number of allocated TLS items.
 * Updated atomically. Update before allocation and after free. */
static unsigned cTLSAllocated;


int     __libc_TLSAlloc(void)
{
    LIBCLOG_ENTER("\n");
    unsigned        cTries;

    /*
     * Space left?
     */
    if (__atomic_increment_max(&cTLSAllocated, __LIBC_TLS_MAX))
    {
        LIBC_ASSERTM_FAILED("Out of TLS entries! cur=%d max=%d\n", cTLSAllocated, __LIBC_TLS_MAX);
        LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Find a free entry (bit).
     * We'll scan the bitmap 32 times before we give up.
     */
    for (cTries = 32; cTries > 0; cTries--)
    {
        /*
         * Scan the bitmap int by int.
         */
        unsigned *pu = &auBitmap[0];
        while (pu < &auBitmap[sizeof(auBitmap) / sizeof(auBitmap[0])])
        {
            if (*pu != ~0)
            {
                /*
                 * Look for free bit.
                 */
                do
                {
                    unsigned uBit;
                    for (uBit = 0; uBit < ENTRY_BITS; uBit++)
                    {
                        if (!__atomic_set_bit(pu, uBit))
                        {
                            int iBitRet = (pu - &auBitmap[0]) * ENTRY_BITS + uBit;
                            if (iBitRet < __LIBC_TLS_MAX)
                                LIBCLOG_RETURN_INT(iBitRet);
                        }
                    }
                } while (*pu != ~0);
            }

            /* next entry */
            pu++;
        }
    }

    LIBC_ASSERTM_FAILED("We're giving up finding a free enter!!! cur=%d max=%d\n", cTLSAllocated, __LIBC_TLS_MAX);
    LIBCLOG_RETURN_INT(-1);
}


int     __libc_TLSFree(int iIndex)
{
    LIBCLOG_ENTER("iIndex=%d\n", iIndex);
    /*
     * Validate index
     */
    if (    iIndex < 0
        ||  iIndex >= __LIBC_TLS_MAX
        ||  __atomic_test_bit(&auBitmap[0], iIndex)
            )
    {
        LIBC_ASSERTM_FAILED("Bad index %d. (max=%d)\n", iIndex, __LIBC_TLS_MAX);
        errno = EINVAL;
        LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Clear the bitmap bit.
     */
    __atomic_clear_bit(&auBitmap[0], iIndex);

    /*
     * Decrement the allocation count.
     */
    LIBC_ASSERT(cTLSAllocated > 0);
    __atomic_decrement(&cTLSAllocated);
    LIBCLOG_RETURN_INT(0);
}


void *  __libc_TLSGet(int iIndex)
{
    LIBCLOG_ENTER("iIndex=%d\n", iIndex);
    /*
     * Validate index
     */
    if (    iIndex < 0
        ||  iIndex >= __LIBC_TLS_MAX
        ||  __atomic_test_bit(&auBitmap[0], iIndex)
            )
    {
        LIBC_ASSERTM_FAILED("Bad index %d. (max=%d)\n", iIndex, __LIBC_TLS_MAX);
        errno = EINVAL;
        LIBCLOG_RETURN_P(NULL);
    }

    /*
     * Get it.
     */
    LIBCLOG_RETURN_P(__libc_threadCurrent()->apvTLS[iIndex]);
}


int     __libc_TLSSet(int iIndex, void *pvValue)
{
    LIBCLOG_ENTER("iIndex=%d pvValue=%p\n", iIndex, pvValue);
    /*
     * Validate index
     */
    if (    iIndex < 0
        ||  iIndex >= __LIBC_TLS_MAX
        ||  __atomic_test_bit(&auBitmap[0], iIndex)
            )
    {
        LIBC_ASSERTM_FAILED("Bad index %d. (max=%d)\n", iIndex, __LIBC_TLS_MAX);
        errno = EINVAL;
        LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Set it.
     */
    __libc_threadCurrent()->apvTLS[iIndex] = pvValue;
    LIBCLOG_RETURN_INT(0);
}

