/* $Id: socketops.c 1289 2004-03-15 01:50:57Z bird $ */
/** @file
 *
 * socket handle operations.
 *
 * Copyright (c) 2003 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of Innotek LIBC.
 *
 * Innotek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Innotek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Innotek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#define INCL_TCPIP_ALLIOCTLS
#include "libc-alias.h"
#include <errno.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <netinet/ip.h>
#include <netinet/ip_var.h>
#include <netinet/ip_icmp.h>
#include <netinet/tcp.h>
#include <netinet/tcp_var.h>
#include <netinet/udp.h>
#include <netinet/udp_var.h>
#include <net/route.h>

#include <sys/fcntl.h>
#include <sys/filio.h>
#include <emx/io.h>
#define __LIBC_LOG_GROUP __LIBC_LOG_GRP_SOCKET
#include <InnoTekLIBC/logstrict.h>
#include "socket.h"


/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
#ifdef TCPV40HDRS
/* not included in the TCPV40HDRS - this is from the BSD4.4 stack headers
                                    and doesn't have to right!!! */
struct arpreq_tr {
        struct  sockaddr arp_pa;                /* protocol address */
        struct  sockaddr arp_ha;                /* hardware address */
        long_int arp_flags;                     /* flags */
        u_short arp_rcf;                        /* token ring routing control field */
        u_short arp_rseg[8];                    /* token ring routing segments */
};

#define MAX_IN_MULTI      16*IP_MAX_MEMBERSHIPS      /* 320 max per os2 */
struct  addrreq  {                              /* get multicast addresses */
        char    ifr_name[IFNAMSIZ];
        struct  sockaddr ifr_addrs;
        u_long  maddr[MAX_IN_MULTI];
};

#pragma pack(1)
struct  oarptab {
        struct  in_addr at_iaddr;       /* internet address */
        u_char  at_enaddr[6];           /* ethernet address */
        u_char  at_timer;               /* minutes since last reference */
        u_char  at_flags;               /* flags */
#ifdef KERNEL
        struct  mbuf *at_hold;          /* last packet until resolved/timeout */
#else
        void * at_hold;
#endif
        u_short at_rcf;                 /* token ring routing control field */
        u_short at_rseg[8];             /* token ring routing segments */
        u_long  at_millisec;            /* TOD milliseconds of last update */
        u_short at_interface;           /* interface index */
};
#pragma pack()

#pragma pack(1)
struct  statatreq {
        u_long addr;
        short interface;
        u_long mask;
        u_long broadcast;
};
#pragma pack()

#endif

/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int __libsocket_ops_Close(PLIBCFH pFH, int fh);
static int __libsocket_ops_Read(PLIBCFH pFH, int fh, void *pvBuf, size_t cbRead, size_t *pcbRead);
static int __libsocket_ops_Write(PLIBCFH pFH, int fh, const void *pvBuf, size_t cbWrite, size_t *pcbWritten);
static int __libsocket_ops_Duplicate(PLIBCFH pFH, int fh, int *pfhNew);
static int __libsocket_ops_FileControl(PLIBCFH pFH, int fh, int iIOControl, int iArg, int *prc);
static int __libsocket_ops_IOControl(PLIBCFH pFH, int fh, int iIOControl, int iArg, int *prc);
static int __libsocket_ops_Select(int cFHs, struct fd_set *pRead, struct fd_set *pWrite, struct fd_set *pExcept, struct timeval *tv, int *prc);


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** Operations on sockets. */
LIBCFHOPS __libsocket_gSocketOps =
{
    enmFH_Socket, /* Handle type. */
    __libsocket_ops_Close,
    __libsocket_ops_Read,
    __libsocket_ops_Write,
    __libsocket_ops_Duplicate,
    __libsocket_ops_FileControl,
    __libsocket_ops_IOControl,
    __libsocket_ops_Select
};


/** Close operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 */
static int __libsocket_ops_Close(PLIBCFH pFH, int fh)
{
    LIBCLOG_ENTER("pFH=%p fh=%d\n", pFH, fh);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int             rc;
    rc = __libsocket_soclose(pSocketFH->iSocket);
    if (rc)
    {
        rc = -__libsocket_getSocketErrno();
        LIBCLOG_RETURN_INT(rc);
    }
    LIBCLOG_RETURN_INT(0);
}


/** Read operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pvBuf       Pointer to the buffer to read into.
 * @param   cbRead      Number of bytes to read.
 * @param   pcbRead     Where to store the count of bytes actually read.
 */
static int __libsocket_ops_Read(PLIBCFH pFH, int fh, void *pvBuf, size_t cbRead, size_t *pcbRead)
{
    LIBCLOG_ENTER("pFH=%p fh=%d pvBuf=%p cbRead=%d pcbRead=%p\n", pFH, fh, pvBuf, cbRead, pcbRead);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int             rc;

    rc = __libsocket_recv(pSocketFH->iSocket, pvBuf, cbRead, 0);
    if (rc < 0)
    {
        *pcbRead = 0;
        rc = -__libsocket_getSocketErrno();
        LIBCLOG_RETURN_INT(rc);
    }

    *pcbRead = rc;
    LIBCLOG_RETURN_INT(0);
}

/** Write operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pvBuf       Pointer to the buffer which contains the data to write.
 * @param   cbWrite     Number of bytes to write.
 * @param   pcbWritten  Where to store the count of bytes actually written.
 */
static int __libsocket_ops_Write(PLIBCFH pFH, int fh, const void *pvBuf, size_t cbWrite, size_t *pcbWritten)
{
    LIBCLOG_ENTER("pFH=%p fh=%d pvBuf=%p cbWrite=%d pcbWritten=%p\n", pFH, fh, pvBuf, cbWrite, pcbWritten);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    int             rc;

    rc = __libsocket_send(pSocketFH->iSocket, pvBuf, cbWrite, 0);
    if (rc < 0)
    {
        *pcbWritten = 0;
        rc = -__libsocket_getSocketErrno();
        LIBCLOG_RETURN_INT(rc);
    }

    *pcbWritten = rc;
    LIBCLOG_RETURN_INT(0);
}


/** Duplicate handle operation.
 * @returns 0 on success, OS/2 error code on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   pfhNew      Where to store the duplicate filehandle.
 *                      The input value describe how the handle is to be
 *                      duplicated. If it's -1 a new handle is allocated.
 *                      Any other value will result in that value to be
 *                      used as handle. Any existing handle with that
 *                      value will be closed.
 */
static int __libsocket_ops_Duplicate(PLIBCFH pFH, int fh, int *pfhNew)
{
    LIBCLOG_ENTER("pFH=%p fh=%d pfhNew=%p\n", pFH, fh, pfhNew);
    LIBCLOG_RETURN_INT(-ENOSYS);
}


/** File Control operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   iIOControl  Which file file control operation to perform.
 * @param   iArg        Argument which content is specific to each
 *                      iIOControl operation.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int __libsocket_ops_FileControl(PLIBCFH pFH, int fh, int iIOControl, int iArg, int *prc)
{
    LIBCLOG_ENTER("pFH=%p fh=%d iIOControl=%#x iArg=%#x prc=%p\n", pFH, fh, iIOControl, iArg, prc);
    /*PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;*/
    LIBCLOG_RETURN_INT(-ENOSYS);
}


/** I/O Control operation.
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   pFH         Pointer to the handle structure to operate on.
 * @param   fh          It's associated filehandle.
 * @param   iIOControl  Which I/O control operation to perform.
 * @param   iArg        Argument which content is specific to each
 *                      iIOControl operation.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int __libsocket_ops_IOControl(PLIBCFH pFH, int fh, int iIOControl, int iArg, int *prc)
{
    LIBCLOG_ENTER("pFH=%p fh=%d iIOControl=%#x iArg=%#x prc=%p\n", pFH, fh, iIOControl, iArg, prc);
    PLIBCSOCKETFH   pSocketFH = (PLIBCSOCKETFH)pFH;
    char           *pchArg = (char *)iArg;
    int             rc;
#ifndef TCPV40HDRS
    switch (__IOCLW(iIOControl))
    {
        case SIOSTATARP:
            rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct oarptab));
            break;
        case SIOSTATAT:
            /** this isn't really suitable for this ioctl interface!! */
            rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct statatreq) + 2);
            break;
        case SIOSTATIF:
            rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct ifmib));
            break;
        case SIOSTATIF42:
            /* What the h*** is the difference between the SIOSTATIF42 ioctl and SIOSTATIF?
               The docs doesn't make sense when looking in the headers... */
            rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct ifmib));
            break;
        case SIOSTATRT:
            rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct rtentries));
            break;
        case SIOSTATSO:
            /** this isn't really suitable for this ioctl interface!! */
            rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, sizeof(struct sockaddr));
            break;

        case FIONBIO:
            rc = __libsocket_ioctl(pSocketFH->iSocket, iIOControl, pchArg);
            if (rc)
            {
                if (*(unsigned*)pchArg)
                    pSocketFH->core.fFlags |= O_NDELAY;
                else
                    pSocketFH->core.fFlags &= ~O_NDELAY;
            }
            break;

        default:
            rc = __libsocket_ioctl(pSocketFH->iSocket, iIOControl, pchArg);
            break;
    }

#else
    int     cb;

    /*
     * This ioctl interface requires a size, which thus is something we
     * must determin. :-/
     */
    switch (__IOCLW(iIOControl))
    {
        case __IOCLW(FIONREAD):
        case __IOCLW(FIOASYNC):
        case __IOCLW(SIOCATMARK):
        case __IOCLW(FIONBIO):
        case __IOCLW(SIOCSHIWAT):
        case __IOCLW(SIOCGHIWAT):
        case __IOCLW(SIOCSLOWAT):
        case __IOCLW(SIOCGLOWAT):
        case __IOCLW(SIOCSPGRP):
        case __IOCLW(SIOCGPGRP):
            cb = sizeof(int);
            break;

        case __IOCLW(SIOCSHOSTID):
        case __IOCLW(SIOCGNBNAME):
        case __IOCLW(SIOCSNBNAME):
        case __IOCLW(SIOCGNCBFN):
            cb = sizeof(long);
            break;

        case __IOCLW(FIONSTATUS):
            cb = sizeof(short) * 4;
            break;

        case __IOCLW(FIOBSTATUS):
            cb = sizeof(short);
            break;

        case __IOCLW(SIOCGIFADDR):
        case __IOCLW(SIOCGIFBRDADDR):
        case __IOCLW(SIOCGIFDSTADDR):
        case __IOCLW(SIOCGIFFLAGS):
        case __IOCLW(SIOCGIFMETRIC):
        case __IOCLW(SIOCGIFNETMASK):
        case __IOCLW(SIOCSIFADDR):
        case __IOCLW(SIOCSIFBRDADDR):
        case __IOCLW(SIOCSIFDSTADDR):
        case __IOCLW(SIOCSIFFLAGS):
        case __IOCLW(SIOCSIFMETRIC):
        case __IOCLW(SIOCSIFNETMASK):
        case __IOCLW(SIOCSIFALLRTB):
        case __IOCLW(SIOCSIF802_3):
        case __IOCLW(SIOCSIFNO802_3):
        case __IOCLW(SIOCSIFNOREDIR):
        case __IOCLW(SIOCSIFYESREDIR):
        case __IOCLW(SIOCSIFMTU):
        case __IOCLW(SIOCSIFFDDI):
        case __IOCLW(SIOCSIFNOFDDI):
        case __IOCLW(SIOCSRDBRD):
        case __IOCLW(SIOCADDMULTI):
        case __IOCLW(SIOCDELMULTI):
        case __IOCLW(SIOCMULTISBC):
        case __IOCLW(SIOCMULTISFA):
        case __IOCLW(SIOCDIFADDR):
        case __IOCLW(SIOCGUNIT):
        case __IOCLW(SIOCGIFVALID):
        case __IOCLW(SIOCGIFEFLAGS):
        case __IOCLW(SIOCSIFEFLAGS):
        case __IOCLW(SIOCGIFTRACE):
        case __IOCLW(SIOCSIFTRACE):
        case __IOCLW(SIOCSSTAT):
        case __IOCLW(SIOCGSTAT):
            cb = sizeof(struct ifreq);
            break;

        case __IOCLW(SIOCADDRT):
        case __IOCLW(SIOCDELRT):
            cb = sizeof(struct rtentry);
            break;

        case __IOCLW(SIOCSARP):
        case __IOCLW(SIOCGARP):
        case __IOCLW(SIOCDARP):
            cb = sizeof(struct arpreq);
            break;

        case __IOCLW(SIOCGIFCONF):
            cb = sizeof(struct ifconf);
            break;

        case __IOCLW(SIOCSARP_TR):
        case __IOCLW(SIOCGARP_TR):
            cb = sizeof(struct arpreq_tr);
            break;

        /* duplicate! Bug in BSD 4.3 stack I'd say.
        case __IOCLW(SIOCAIFADDR):
            cb = sizeof(struct ifaliasreq);
            break;
         */

        case __IOCLW(SIOCGIFBOUND):
            cb = sizeof(struct bndreq);
            break;

        case __IOCLW(SIOCGMCAST):
            cb = sizeof(struct addrreq);
            break;

        case __IOCLW(SIOSTATMBUF):
            cb = sizeof(struct mbstat);
            break;

        case __IOCLW(SIOSTATTCP):
            cb = sizeof(struct tcpstat);
            break;

        case __IOCLW(SIOSTATUDP):
            cb = sizeof(struct udpstat);
            break;

        case __IOCLW(SIOSTATIP):
            cb = sizeof(struct ipstat);
            break;

        case __IOCLW(SIOSTATSO):
            cb = sizeof(struct sockaddr);
            break;

        case __IOCLW(SIOSTATRT):
            cb = sizeof(struct rtentries);
            break;

        case __IOCLW(SIOFLUSHRT):
            cb = sizeof(long);
            break;

        case __IOCLW(SIOSTATICMP):
            cb = sizeof(struct icmpstat);
            break;

        case __IOCLW(SIOSTATIF):
            cb = sizeof(struct ifmib);
            break;

        case __IOCLW(SIOSTATAT):
            /** this isn't really suitable for this ioctl interface!! */
            cb = sizeof(struct statatreq) + 2;
            break;

        case __IOCLW(SIOSTATARP):
            cb = sizeof(struct oarptab);
            break;

        case __IOCLW(SIOSTATIF42):
            cb = sizeof(struct ifmib); /* docs are unclear here */
            break;

        case __IOCLW(SIOSTATCNTRT):
            cb = sizeof(int);
            break;

        case __IOCLW(SIOSTATCNTAT):
            cb = sizeof(int);
            break;

        default:
            cb = -1;
            break;

/** @todo There is a bunch of IOCtls in the BSD 4.3 stack
 *        for which I don't have the docs and don't know exact
 *        what sizes to use:
 *               FIOTCPCKSUM     ioc('f', 128)
 *               FIONURG         ioc('f', 121)
 *               SIOMETRIC1RT    ioc('r', 12)
 *               SIOMETRIC2RT    ioc('r', 13)
 *               SIOMETRIC3RT    ioc('r', 14)
 *               SIOMETRIC4RT    ioc('r', 15)
 *               SIOCREGADDNET   ioc('r', 12)
 *               SIOCREGDELNET   ioc('r', 13)
 *               SIOCREGROUTES   ioc('r', 14)
 *               SIOCFLUSHROUTES ioc('r', 15)
 *               SIOCSIFSETSIG   ioc('i', 25)
 *               SIOCSIFCLRSIG   ioc('i', 26)
 *               SIOCSIFBRD      ioc('i', 27)
 *               SIOCGIFLOAD     ioc('i', 27)
 *               SIOCSIFFILTERSRC ioc('i', 28)
 *               SIOCGIFFILTERSRC ioc('i',29)
 *               SIOCSIFSNMPSIG  ioc('i', 33)
 *               SIOCSIFSNMPCLR  ioc('i', 34)
 *               SIOCSIFSNMPCRC  ioc('i', 35)
 *               SIOCSIFPRIORITY ioc('i', 36)
 *               SIOCGIFPRIORITY ioc('i', 37)
 *               SIOCSIFFILTERDST ioc('i', 38)
 *               SIOCGIFFILTERDST ioc('i',39)
 *               SIOCSIFSPIPE     ioc('i',71)
 *               SIOCSIFRPIPE     ioc('i',72)
 *               SIOCSIFTCPSEG   ioc('i',73)
 *               SIOCSIFUSE576   ioc('i',74)
 */
    } /* size switch */

    /*
     * Do the call.
     */
    if (cb >= 0)
    {
        switch (__IOCLW(iIOControl))
        {
            case FIONBIO:
                rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, cb);
                if (rc)
                {
                    if (*(unsigned*)pchArg)
                        pSocketFH->core.fFlags |= O_NDELAY;
                    else
                        pSocketFH->core.fFlags &= ~O_NDELAY;
                }
                break;

            default:
                rc = __libsocket_os2_ioctl(pSocketFH->iSocket, iIOControl, pchArg, cb);
                break;
        }
    }
    else
    {   /*  */
        *prc = -1;
        LIBCLOG_RETURN_INT(-ENOSYS);
    }

#endif

    *prc = rc;
    if (rc < 0)
    {
        rc = -__libsocket_getSocketErrno();
        LIBCLOG_RETURN_INT(rc);
    }
    LIBCLOG_RETURN_INT(0);
}


/** Select operation.
 * The select operation is only performed if all handles have the same
 * select routine (the main worker checks this).
 *
 * @returns 0 on success.
 * @returns OS/2 error code or negated errno on failure.
 * @param   cFHs        Range of handles to be tested.
 * @param   pRead       Bitmap for file handles to wait upon to become ready for reading.
 * @param   pWrite      Bitmap for file handles to wait upon to become ready for writing.
 * @param   pExcept     Bitmap of file handles to wait on (error) exceptions from.
 * @param   tv          Timeout value.
 * @param   prc         Where to store the value which upon success is
 *                      returned to the caller.
 */
static int __libsocket_ops_Select(int cFHs, struct fd_set *pRead, struct fd_set *pWrite, struct fd_set *pExcept, struct timeval *tv, int *prc)
{
    LIBCLOG_ENTER("cFHs=%d pRead=%p pWrite=%p pExcept=%p tv=%p prc=%d\n",
                  cFHs, pRead, pWrite, pExcept, tv, prc);
    int rc;
    int saved_errno = errno;
    rc = bsdselect(cFHs, pRead, pWrite, pExcept, tv);
    *prc = rc;
    rc = rc >= 0 ? 0 : -errno;
    errno = saved_errno;
    LIBCLOG_RETURN_INT(rc);
}

