/* $Id: thread.h 1287 2004-03-11 03:05:55Z bird $ */
/** @file
 *
 * LIBC Thread Handling.
 *
 * Copyright (c) 2004 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of InnoTek LIBC.
 *
 * InnoTek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * InnoTek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with InnoTek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef __InnoTekLIBC_thread_h__
#define __InnoTekLIBC_thread_h__

/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
/** Maximum number of TLS variables supported by LIBC.
 * The current limit (128) is higher than for instance WinNT. But if
 * you think we're wasting space, look at the page padding of the thread
 * structure... */
#define __LIBC_TLS_MAX      128


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <sys/cdefs.h>
#include <time.h>                       /* struct tm; */
#include <signal.h>


/*******************************************************************************
*   Structures and Typedefs                                                    *
*******************************************************************************/
struct _uheap;

/**
 * Per thread structure for LIBC.
 *
 * This structure contains buffers and variables which in a single threaded
 * LIBC would be static.
 *
 * Members most frequently used have been put together at the front.
 */
typedef struct __libc_threadCurrentSlow
{
    /** errno value. (Comes first, see errnofun.s.) */
    int             iErrNo;
    /** Default tiled heap. */
    struct _uheap * pTiledHeap;
    /** Default regular heap. */
    struct _uheap * pRegularHeap;
    /** Old TLS variable. */
    void           *pvThreadStoreVar;
    /** New TLS variable array. */
    void           *apvTLS[__LIBC_TLS_MAX];

    /** Current rand() seed value. */
    unsigned int    iRand;
    /** Buffer used by asctime() and indirectly by ctime() . (Adds two 2 bytes for padding). */
    char            szAscTimeAndCTimeBuf[26+2];
    /** Buffer used by gmtime() and localtime(). */
    struct tm       GmTimeAndLocalTimeBuf;
    /** Buffer used by tmpnam(). */
    char            szTmpNamBuf[16];
    /** Current posistion of strtok(). */
    unsigned char  *pszStrTokPos;
    /** Buffer used by strerror() to format unknown error values. */
    char            szStrErrorBuf[28];
    /** Buffer used by _getvol(). */
    char            szVolLabelBuf[12];
    /** Buffer used by ttyname(). */
    char            szTTYNameBuf[32];

    /** Data used by the backends. */
    union __libc_backend_data
    {
        struct __libc_sys
        {
            /** Blocked signal mask. */
            sigset_t            sig_blocked;
            /** Pending signal mask. */
            sigset_t            sig_pending;
            /** Signal actions. */
            struct sigaction    signals[NSIG];

            /** Directory find data entry.
             * Used by __findfirst() and __findnext(). */
            struct find_data
            {
                /** Directory handle. HDIR_CREATE if no session opened. */
                unsigned long   hdir;
                /** Type of buffer content. FIL_STANDARDL or FIL_STANDARD,
                 * i.e. FILEFINDBUF3 or FILEFINDBUF3L. */
                unsigned long   fType;
                /** Number of files left in the buffer. */
                unsigned long   cFiles;
                /** Pointer to the next entry. Don't test on this, test on cFiles! */
                const char     *pchNext;
                /** Buffer. */
                char            achBuffer[2048];
            } fd;
        } sys;
    } b;

    /** Data used in special periods of a threads life. */
    union
    {
        struct __libc_thread_startup
        {
            /** Thread argument. */
            void    *pvArg;
            /** Thread routine. */
            void   (*pfnStart)(void *pvArg);
        } startup;
    } u;
} __LIBC_THREAD;

#ifndef __LIBC_THREAD_DECLARED
#define __LIBC_THREAD_DECLARED
typedef struct __libc_threadCurrentSlow *__LIBC_PTHREAD, **__LIBC_PPTHREAD;
#endif


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
__BEGIN_DECLS
/** Pointer to the TLS ULONG (OS/2 rules) which will point to the LIBC thread
 * structure for the current thread.
 * The TLS ULONG is allocated by __init_dll(). The thread structure it points to
 * is allocated on demand. */
extern __LIBC_PPTHREAD  __libc_gpTLS;
__END_DECLS


/*******************************************************************************
*   External Functions                                                         *
*******************************************************************************/
__BEGIN_DECLS
/**
 * Get the thread structure for the current thread.
 *
 * Will automatically allocate a thread structure if such is not yet done
 * for the thread.
 *
 * @returns pointer to current thread struct.

 * @remark  This API is considered to be internal to LIBC and is thus not
 *          exposed in the shared library version of LIBC. Please don't call it.
 *          External LIBs should use the __libc_TLS*() API.
 */
#define __libc_threadCurrent()   (*__libc_gpTLS ? *__libc_gpTLS : __libc_threadCurrentSlow())


/**
 * Get the thread structure for the current thread.
 *
 * Used by the __libc_threadCurrent() macro for allocating a thread structure for the
 * current thread when such doesn't exist.
 *
 * @returns pointer to current thread struct.

 * @remark  This API is considered to be internal to LIBC and is thus not
 *          exposed in the shared library version of LIBC. Please don't call it.
 *          External LIBs should use the __libc_TLS*() API.
 */
__LIBC_PTHREAD __libc_threadCurrentSlow(void);


/**
 * Get the thread structure for the current thread.
 *
 * Do not create anything automatically.
 *
 * @returns pointer to current thread struct.
 * @returns NULL if not initiated.
 *
 * @remark  This API is considered to be internal to LIBC and is thus not
 *          exposed in the shared library version of LIBC. Please don't call it.
 *          External LIBs should use the __libc_TLS*() API.
 */
#define __libc_threadCurrentNoAuto()   (__libc_gpTLS ? *__libc_gpTLS : NULL)


/**
 * Allocate and initialize a thread structure for a thread which is yet
 * to be created.
 *
 * @returns Pointer to thread structure.
 * @returns NULL on error. errno set.
 */
__LIBC_PTHREAD __libc_threadAlloc(void);


/**
 * Free a thread structure.
 *
 * @param   pThrd   Pointer to the thread structure to free.
 *                  Must be valid.
 * @remark  If pThrd is for the current thread the thread must be
 *          in the very final termination stage.
 */
void    __libc_threadFree(__LIBC_PTHREAD pThrd);


/** @group InnoTek LIBC Thread Local Storage
 * @{
 */

/**
 * Allocates a TLS entry.
 *
 * @returns index of the allocated TLS index.
 * @returns -1 on failure. errno set.
 */
int     __libc_TLSAlloc(void);

/**
 * Frees a TLS entry allocated by __libc_TLSAlloc().
 *
 * @returns 0 on success.
 * @returns -1 on failure. errno set.
 * @param   iTLSIndex   Value returned by __libc_TLSAlloc().
 */
int     __libc_TLSFree(int iIndex);

/**
 * Get the value stored in an allocated TLS entry.
 *
 * @returns value in given TLS entry.
 * @returns NULL on failure with errno set.
 * @param   iTLSIndex   Value returned by __libc_TLSAlloc().
 */
void *  __libc_TLSGet(int iIndex);

/**
 * Set the value stored in an allocated TLS entry.
 *
 * @returns 0 on success.
 * @returns -1 on failure. errno set.
 * @param   iTLSIndex   Value returned by __libc_TLSAlloc().
 * @param   pvValue     Value to store.
 */
int     __libc_TLSSet(int iIndex, void *pvValue);

/** @} */
__END_DECLS

#endif
