/* $Id: usleep.c 1270 2004-02-20 12:03:42Z bird $ */
/** @file
 *
 * usleep().
 *
 * Copyright (c) 2004 InnoTek Systemberatung GmbH
 * Author: knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of Innotek LIBC.
 *
 * Innotek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Innotek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Innotek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#include <unistd.h>
#include <errno.h>
#define INCL_BASE
#define INCL_FSMACROS
#include <os2.h>
#include <emx/libclog.h>


/**
 * Suspend execution of the current thread for a given number of microseconds
 * or till a signal is received.
 *
 * @returns 0 on success.
 * @returns -1 if useconds is equal or larger than one million.
 * @param   useconds    Number of microseconds to sleep.
 *                      If 0 nothing is done.
 *                      If equal or larger than one million errno is set to EINVAL and
 *                      returning -1.
 * @author  knut st. osmundsen <bird-srcspam@anduin.net>
 * @remark  For relativly small sleeps this api temporarily changes the thread
 *          priority to timecritical (that is, if it's in the normal or idle priority
 *          classes) to increase precision. This means that if a signal or other
 *          asyncronous event is executed, it will be executed at wrong priority.
 *          It also means that if such code changes the priority it will be undone.
 * @todo    Check out signal effects mentioned in the SUS function description.
 */
int _STD(usleep)(useconds_t useconds)
{
    LIBCLOG_ENTER("useconds=%d", useconds);
    ULONG   msSleep;
    FS_VAR();

    /*
     * Check input.
     */
    if (useconds >= 1000000)
    {
        errno = EINVAL;
        LIBCLOG_RETURN_INT(-1);
    }

    /*
     * If zero, nothing to do.
     */
    if (useconds == 0)
        LIBCLOG_RETURN_INT(0);

    /*
     * Calc millisecond wait.
     */
    msSleep = (useconds + 999) / 1000;

    FS_SAVE();

    /*
     * For small sleeps, precision might be important. OS/2 have a very bad
     * precision on it's DosSleep api, the accuracy can easily be off by a
     * timeslice. * It might be bad to return 32milliseconds late for small
     * sleeps, thus we do a little hack for values of two timeslices or less.
     * Timecritical threads are scheduled more precisely thank normal and idle
     * threads, so we temporarily change the priority of the thread while
     * waiting.
     */
    if (useconds < 65000)
    {
        PTIB    pTib;
        PPIB    pPib;
        ULONG   ulOldPri;

        DosGetInfoBlocks(&pTib, &pPib);
        ulOldPri = pTib->tib_ptib2->tib2_ulpri;

        if (ulOldPri < (PRTYC_TIMECRITICAL << 8))
        {
            LIBCLOG_MSG("current priority %#05x temporarily boosting priority\n", ulOldPri);
            DosSetPriority(PRTYS_THREAD, PRTYC_TIMECRITICAL, 0, 0);
        }

        /*
         * For some reason we're frequently timing out too early the range
         * 30-32ms. We're compensating here, just to make sure we're never
         * too early without reason. (bird lives by that policy.)
         */
        if (msSleep >= 30 && msSleep < 33)
            msSleep = 33;
        LIBCLOG_MSG("DosSleep(%d)\n", msSleep);
        DosSleep(msSleep);

        if (ulOldPri < (PRTYC_TIMECRITICAL << 8))
        {
            APIRET rc = DosSetPriority(PRTYS_THREAD, ulOldPri >> 8, ulOldPri & 0xff, 0);
            if (rc)
            {
                LIBCLOG_MSG("DosSetPriority(,%x,%x,0) failed with rc=%d\n",
                            ulOldPri >> 8, ulOldPri & 0xff, rc);
                //@todo add assertion.
                DosSetPriority(PRTYS_PROCESS, PRTYC_REGULAR, 0, 0);
            }
        }
    }
    else
    {
        /*
         * Simple DosSleep().
         */
        LIBCLOG_MSG("DosSleep(%d)\n", msSleep);
        DosSleep(msSleep);
    }

    FS_RESTORE();
    return 0;
}
