/* _vsopen.c (emx+gcc) -- Copyright (c) 1990-1996 by Eberhard Mattes */

#include "libc-alias.h"
#include <stdarg.h>
#include <io.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <emx/io.h>
#include <emx/syscalls.h>

/* Bugs: O_TRUNC|O_RDONLY not implemented */
/*       O_TEXT|O_WRONLY  does/can not overwrite Ctrl-Z */

int _fmode_bin = 0;                     /* Set non-zero to make binary mode default */

#define SH_MASK           0x70

/**
 * This function may be called as follows (assuming that fOpen does
 *  not include O_CREAT nor O_SIZE):
 *
 *  _vsopen(pszName, fOpen, fShare)
 *  _vsopen(pszName, fOpen | O_CREAT, fShare, pmode)
 *  _vsopen(pszName, fOpen | O_SIZE, fShare, off_t size)
 *  _vsopen(pszName, fOpen | O_CREAT | O_SIZE, fShare, pmode, off_t size)
 *
 * @remark O_SIZE means a off_t sized argument not unsigned long as emxlib say.
 */
int _vsopen(const char *pszName, int fOpen, int fShare, va_list va)
{
    int     hFile;
    /** Flags is for __open.
     *  0 -  1  Access mode (O_RDONLY, O_WRONLY, O_RDWR).
     *  4 -  6  Shared (SH_*)
     *  8 - 15  File attributes (DOS style).
     * 16 - 23  __open flags, _SO_*
     */
    int         fFlags;
    unsigned    fLibc;                  /* LIBC handle flags. */
    int         saved_errno;
    char        chDummy;
    int         fCtrlZKludge = 0;
    off_t       cbInitial;
    PLIBCFH     pFH;

    /*
     * Validate input
     */
    if ((fOpen & O_ACCMODE) == O_RDONLY && (fOpen & (O_TRUNC|O_CREAT)))
    {
        errno = EINVAL;
        return -1;
    }

    /*
     * Build fFlags.
     */
    fLibc = fOpen & (O_ACCMODE|O_NDELAY|O_APPEND|O_SYNC/*|O_DIRECT*/|O_NOINHERIT);
    if (   !(fOpen & O_BINARY)
        && ((fOpen & O_TEXT) || !_fmode_bin))
        fLibc |= O_TEXT;

    if ((fLibc & O_TEXT) && (fOpen & O_APPEND) && (fOpen & O_ACCMODE) == O_WRONLY)
    {
        /* The caller requests to open a text file for appending in
           write-only.  To remove the Ctrl-Z (if there is any), we have
           to temporarily open the file in read/write mode. */

        fFlags = O_RDWR | (fShare & SH_MASK);
        fCtrlZKludge = 1;
    }
    else
        fFlags = (fOpen & O_ACCMODE) | (fShare & SH_MASK);

    /* pmode(=attr) - attr is 2nd byte of flags. */
    if (fOpen & O_CREAT)
    {
        int attr, pmode;

        attr = 0;
        pmode = va_arg(va, int);
        if (!(pmode & S_IWRITE))
            attr |= _A_RDONLY;
        fFlags |= (attr << 8) | _SO_CREAT;
        if (fOpen & O_EXCL)
            fFlags |= _SO_EXCL;
    }

    if (fOpen & O_TRUNC)
        fFlags |= _SO_TRUNC;

    if (fOpen & O_NOINHERIT)
        fFlags |= _SO_NOINHERIT;

    if (fOpen & O_SYNC)
        fFlags |= _SO_SYNC;

    /* Initial file size */
    cbInitial = 0;
    if (fOpen & O_SIZE)
    {
        cbInitial = va_arg(va, off_t);
        fFlags |= _SO_SIZE;
    }

    /*
     * Open - note Ctrl-Z hack.
     */
    saved_errno = errno;
    hFile = __open(pszName, fFlags, cbInitial, fLibc, &pFH);
    if (hFile < 0 && fCtrlZKludge && errno == EACCES)
    {
        /* Perhaps read access is denied. Try again. */
        errno = saved_errno;
        fCtrlZKludge = 0;
        fFlags = (fFlags & ~O_ACCMODE) | (fOpen & O_ACCMODE);
        hFile = __open(pszName, fFlags, cbInitial, fLibc, &pFH);
    }
    if (hFile < 0)
        return -1;
    fLibc = pFH->fFlags;                /* get the updated flags. */

    /*
     * Check what we got a handle to.
     */
    if ((fLibc & F_TYPEMASK) == F_DEV)
        fOpen &= ~O_APPEND;

    /*
     * For text files we shall remove eventual ending Ctrl-Z.
     */
    if (    (fLibc & F_TYPEMASK) != F_DEV
        &&  (fLibc & O_TEXT))
    {
        off_t cbSize;

        /* Remove the last character of the file if it is a Ctrl-Z.  Do
           this even if O_SIZE is given as we don't know whether the
           file existed before; removing a spurious Ctrl-Z due to O_SIZE
           should be harmless. */

        cbSize = __lseek(hFile, -1, SEEK_END);
        if (cbSize >= 0)
        {
            if (__read(hFile, &chDummy, 1) == 1 && chDummy == 0x1a)
                __ftruncate(hFile, cbSize); /* Remove Ctrl-Z */
            __lseek(hFile, 0L, SEEK_SET);
        }
    }

    /*
     * Reopen the file in write-only mode if Ctrl-Z hack applied.
     */
    if (fCtrlZKludge)
    {
        __close(hFile);
        fFlags = (fFlags & ~O_ACCMODE) | (fOpen & O_ACCMODE);
        fFlags &= ~_SO_EXCL;       /* Ignore O_EXCL */
        hFile = __open(pszName, fFlags, cbInitial, fLibc & ~F_TYPEMASK, &pFH);
        if (hFile < 0)
            return -1;
    }

    /*
     * When opening a file for appending, move to the end of the file.
     * This is required for passing the file to a child process.
     */
    if (    (fLibc & F_TYPEMASK) == F_FILE
        &&  (fLibc & O_APPEND))
        __lseek(hFile, 0L, SEEK_END);
    errno = saved_errno;

    return hFile;
}
