/* $Id: bsdselect.c 1172 2004-02-04 22:36:47Z bird $ */
/** @file
 *
 * bsdselect().
 *
 * Copyright (c) 2003 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This file is part of Innotek LIBC.
 *
 * Innotek LIBC is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Innotek LIBC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Innotek LIBC; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "libc-alias.h"
#include <errno.h>
#include <sys/socket.h>
#include <sys/fcntl.h>
#include <sys/time.h>
#include <emx/io.h>
#include <emx/libclog.h>
#include "socket.h"


/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#ifdef TCPV40HDRS

#define MY_FD_SET(fd, set)      FD_SET(fd, set)
#define MY_FD_ISSET(fd, set)    FD_ISSET(fd, set)
#define MY_FD_ZERO(set, cb)     memset(set, 0, cb);
#define my_fd_set               fd_set

#else

#define MY_FD_SET(fd, set)      V5_FD_SET(fd, set)
#define	MY_FD_ISSET(fd, set)    V5_FD_ISSET(fd, set)
#define MY_FD_ZERO(set, cb)     memset(set, 0, cb);
#define my_fd_set               v5_fd_set

#endif


/**
 * Calculate the size required for the converted set structure.
 * @returns number of bytes.
 * @param   c       Number of file descriptors specified in the call.
 * @param   pFrom   The select input fd_set structure.
 * @parm    pcFDs   Store the new number of file descriptors (socket handles) to examin.
 */
static inline int calcsize(int c, const struct fd_set *pFrom, int *pcFDs)
{
    int cbRet;
    int i;
#ifdef TCPV40HDRS
    /* here we need to figure out the max real socket handle */
    int iMax;
    for (iMax = *pcFDs - 1, i = 0; i < c; i++)
        if (FD_ISSET(i, pFrom))
        {
            PLIBCSOCKETFH   pFHSocket = __libsocket_FH(i);
            if (pFHSocket && iMax < pFHSocket->iSocket)
                iMax = pFHSocket->iSocket;
        }
    cbRet = (iMax + 8) / 8;             /* iMax is index not count, thus +8 not +7. */
    *pcFDs = iMax + 1;
#else
    for (cbRet = sizeof(u_short), i = 0; i < c; i++)
        if (FD_ISSET(i, pFrom))
            cbRet++;
    *pcFDs = c;
#endif
    if (cbRet < sizeof(struct my_fd_set))
        return sizeof(struct my_fd_set);
    return cbRet;
}

/** Converts the LIBC fd_set strcture pointed to by pFrom with it's LIBC socket handles,
 * to the fd_set strcuture used by the OS/2 tcpip and the OS/2 socket handles.
 * @returns 0 on success.
 * @returns -1 on failure, both errnos set.
 * @param   c       Number of file descriptors specified in the call.
 * @param   cb      Size of pTo. (used for zeroing it)
 * @param   pFrom   The select input fd_set structure.
 * @param   pTo     The structure we present to OS/2 TCPIP.
 *                  This will be initialized.
 * @param   pszType Typestring to use in the log.
 */
static inline int convert(int c, int cb, const struct fd_set *pFrom, struct my_fd_set *pTo, const char *pszType)
{
    int i;
    MY_FD_ZERO(pTo, cb)
    for (i = 0; i < c; i++)
    {
        if (FD_ISSET(i, pFrom))
        {
            PLIBCSOCKETFH   pFHSocket = __libsocket_FH(i);
            if (!pFHSocket)
            {
                LIBCLOG_MSG2("Invalid handle %d specified (%s).\n", i, pszType);
                return -1;
            }
            MY_FD_SET(pFHSocket->iSocket, pTo);
            LIBCLOG_MSG2("%s: %02d -> %03d\n", pszType, i, pFHSocket->iSocket);
        }
    }
    pszType = pszType;
    return 0;
}

/**
 * Updates the pTo structure with the fds marked ready in pFrom.
 *
 * @param   c       Number of file descriptors specified in the call.
 * @param   pFrom   The structure returned from OS/2 TCPIP select.
 * @param   pTo     The structure passed in to select which have to
 *                  be updated for the return.
 * @param   pszType Typestring to use in the log.
 */
static inline void update(int c, const struct my_fd_set *pFrom, struct fd_set *pTo, const char *pszType)
{
    int i;
    for (i = 0; i < c; i++)
    {
        if (FD_ISSET(i, pTo))
        {
            PLIBCSOCKETFH   pFHSocket = __libsocket_FH(i);
            if (pFHSocket)
            {
                if (!MY_FD_ISSET(pFHSocket->iSocket, pFrom))
                {
                    FD_CLR(i, pTo);
                    LIBCLOG_MSG2("%s: %02d -> %03d set\n", pszType, i, pFHSocket->iSocket);
                }
                else
                    LIBCLOG_MSG2("%s: %02d -> %03d clear\n", pszType, i, pFHSocket->iSocket);
            }
        }
    }
    pszType = pszType;
}



int bsdselect(int nfds, struct fd_set *readfds, struct fd_set *writefds, struct fd_set *exceptfds, struct timeval *tv)
{
    LIBCLOG_ENTER("nfds=%d readfds=%p writefds=%p exceptfds=%d tv=%p={tv_sec=%d, rv_usec=%d}\n",
                  nfds, readfds, writefds, exceptfds, tv, tv ? tv->tv_sec : -1, tv ? tv->tv_usec : -1);
#ifdef TCPV40HDRS
    struct fd_set      *pRead;
    struct fd_set      *pWrite;
    struct fd_set      *pExcept;
#else
    struct v5_fd_set   *pRead;
    struct v5_fd_set   *pWrite;
    struct v5_fd_set   *pExcept;
#endif
    int                 rc;
    int                 cb = 0;
    int                 cFDs = 0;

    /*
     * Calculate bitmapsize.
     */
    if (readfds)
        cb = calcsize(nfds, readfds, &cFDs);
    if (writefds)
    {
        int cbT = calcsize(nfds, writefds, &cFDs);
        if (cbT > cb)
            cb = cbT;
    }
    if (exceptfds)
    {
        int cbT = calcsize(nfds, exceptfds, &cFDs);
        if (cbT > cb)
            cb = cbT;
    }

    /*
     * Allocate new bitmaps.
     */
    pRead = NULL;
    if (readfds)
    {
        pRead = alloca(cb);
        if (!pRead)
        {
            __libsocket_setErrno(ENOMEM);
            LIBCLOG_RETURN_INT(-1);
        }
    }

    pWrite = NULL;
    if (writefds)
    {
        pWrite = alloca(cb);
        if (!pWrite)
        {
            __libsocket_setErrno(ENOMEM);
            LIBCLOG_RETURN_INT(-1);
        }
    }

    pExcept = NULL;
    if (exceptfds)
    {
        pExcept = alloca(cb);
        if (!pExcept)
        {
            __libsocket_setErrno(ENOMEM);
            LIBCLOG_RETURN_INT(-1);
        }
    }

    /*
     * Convert the bitmaps.
     */
    if (readfds)
    {
        if (convert(nfds, cb, readfds, pRead, "read "))
            LIBCLOG_RETURN_INT(-1);
    }

    if (writefds)
    {
        if (convert(nfds, cb, writefds, pWrite, "write"))
            LIBCLOG_RETURN_INT(-1);
    }

    if (exceptfds)
    {
        if (convert(nfds, cb, exceptfds, pExcept, "excpt"))
            LIBCLOG_RETURN_INT(-1);
    }

    /*
     * Do the call.
     */
    LIBCLOG_MSG("calling native: cFDs=%d pRead=%p pWrite=%p, pExcept=%p tv=%p\n",
                cFDs, pRead, pWrite, pExcept, tv);
    rc = __libsocket_bsdselect(cFDs, pRead, pWrite, pExcept, tv);
    if (rc < 0)
    {
        __libsocket_setLibcErrno();
        LIBCLOG_RETURN_INT(rc);
    }

    /*
     * Timeout?
     *  Just clear the bitmaps and return.
     */
    if (rc == 0)
    {
        cb = (nfds + 7) / 8;
        if (readfds)
            memset(readfds, 0, cb);
        if (writefds)
            memset(writefds, 0, cb);
        if (exceptfds)
            memset(exceptfds, 0, cb);
        LIBCLOG_RETURN_INT(0);
    }

    /*
     * Convert the bitmaps.
     */
    if (readfds)
        update(nfds, pRead, readfds, "read ");
    if (writefds)
        update(nfds, pWrite, writefds, "write");
    if (exceptfds)
        update(nfds, pExcept, exceptfds, "excpt");

    LIBCLOG_RETURN_INT(rc);
}



