/* $Id: libclog.c 1170 2004-02-04 22:35:57Z bird $ */
/** @file
 *
 * Debug Logging Facility - internal libc use only.
 *
 * InnoTek Systemberatung GmbH confidential
 *
 * Copyright (c) 2004 InnoTek Systemberatung GmbH
 * Author: knut st. osmundsen <bird-srcspam@anduin.net>
 *
 * All Rights Reserved
 *
 */

/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
#define CCHTMPMSGBUFFER 512


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <process.h>
#undef DEBUG_LOGGING
#define DEBUG_LOGGING                   /* need prototypes. */
#include <emx/libclog.h>

#define INCL_BASE
#include <os2.h>


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** Logfile. */
static HFILE    hLog = -1;

/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int openLog(void);
static unsigned getTimestamp(void);
static unsigned getTid(void);


/**
 * Opens the log file.
 * @returns 0 on success.
 * @returns -1 on failure.
 */
static int openLog(void)
{
    char        szFilename[256];
    ULONG       ulAction;
    int         rc;
    char       *pszMsg;

    if ((int)hLog >= 0)
        return 0;

    sprintf(szFilename, "libc_%04x.log", getpid());
    rc = DosOpen(szFilename, &hLog, &ulAction, 0, FILE_NORMAL,
                 OPEN_ACTION_CREATE_IF_NEW | OPEN_ACTION_REPLACE_IF_EXISTS,
                 OPEN_FLAGS_SEQUENTIAL | OPEN_FLAGS_NOINHERIT | OPEN_SHARE_DENYWRITE | OPEN_ACCESS_WRITEONLY,
                 NULL);
    if (rc)
        return -1;

    /* write a log header */
    pszMsg = alloca(CCHTMPMSGBUFFER);
    if (pszMsg)
    {
        PTIB        pTib;
        PPIB        pPib;
        DATETIME    dt;
        const char *psz;
        ULONG       cb;
        int         cch;

        DosGetInfoBlocks(&pTib, &pPib);
        DosGetDateTime(&dt);
        if (DosQueryModuleName(pPib->pib_hmte, sizeof(szFilename), &szFilename[0]))
            szFilename[0] = '\0';

        cch = snprintf(pszMsg, CCHTMPMSGBUFFER,
                       "Opened log at %04d-%02d-%02d %02d:%02d:%02d.%02d\n"
                       "Process ID: %#x (%d) Parent PID: %#x (%d)\n",
                       dt.year, dt.month, dt.day, dt.hours, dt.minutes, dt.seconds, dt.hundredths,
                       (int)pPib->pib_ulpid, (unsigned)pPib->pib_ulpid, (int)pPib->pib_ulppid, (unsigned)pPib->pib_ulppid);
        DosWrite(hLog, pszMsg, cch, &cb);

        cch = snprintf(pszMsg, CCHTMPMSGBUFFER,
                       "Exe hmte  : %#x (%s)\n",
                       (unsigned)pPib->pib_hmte, szFilename);
        DosWrite(hLog, pszMsg, cch, &cb);

        cch = snprintf(pszMsg, CCHTMPMSGBUFFER,
                       "First arg : %s\n"
                       "Second arg: ",
                       pPib->pib_pchcmd);
        DosWrite(hLog, pszMsg, cch, &cb);

        psz = pPib->pib_pchcmd + strlen(pPib->pib_pchcmd) + 1;
        cch = strlen(psz);
        DosWrite(hLog, psz, cch, &cb);
        DosWrite(hLog, "\n", 1, &cb);
    }
    return 0;
}


/**
 * Get ts. (milliseconds preferably)
 */
inline static unsigned getTimestamp(void)
{
    unsigned long ulTs = 0;
    DosQuerySysInfo(QSV_MS_COUNT, QSV_MS_COUNT, &ulTs, sizeof(ulTs));
    return (unsigned)ulTs;
}


/**
 * Gets the current thread id.
 */
inline static unsigned getTid(void)
{
    PTIB    pTib;
    PPIB    pPib;
    DosGetInfoBlocks(&pTib, &pPib);
    return pTib->tib_ptib2->tib2_ultid;
}


/**
 * Output an enter function log message.
 * An enter message is considered to be one line and is appended a newline if
 * none was given.
 *
 * @returns Current timestamp.
 * @param   pszFunction     Name of the function which was entered.
 * @param   pszFormat       Format string to display arguments.
 * @param   ...             Arguments to the format string.
 */
unsigned __libc_LogEnter(const char *pszFunction, const char *pszFormat, ...)
{
    unsigned    uTS = getTimestamp();
    char       *pszMsg;
    int         cch;
    va_list     args;
    ULONG       cb;

    if ((int)hLog < 0 && openLog())
        return 0;

    pszMsg = alloca(CCHTMPMSGBUFFER);
    if (!pszMsg)
        return uTS;

    cch = snprintf(pszMsg, CCHTMPMSGBUFFER, "%08x %02x Entr %s: ", uTS, getTid(), pszFunction);
    va_start(args, pszFormat);
    cch += vsnprintf(pszMsg + cch, CCHTMPMSGBUFFER - cch, pszFormat, args);
    va_end(args);
    if (cch >= CCHTMPMSGBUFFER)
        memcpy(pszMsg + CCHTMPMSGBUFFER - 5, "...\n", 5);
    else if (pszMsg[cch - 1] != '\n')
        memcpy(&pszMsg[cch++], "\n", 2);

    DosWrite(hLog, pszMsg, cch, &cb);
    return uTS;
}

/**
 * Output a leave function log message.
 * A leave message is considered to be one line and is appended a newline if
 * none was given.
 *
 * @param   uEnterTS        The timestamp returned by LogEnter.
 * @param   pszFunction     Name of the function which was entered.
 * @param   pszFormat       Format string to display the result.
 * @param   ...             Arguments to the format string.
 */
void     __libc_LogLeave(unsigned uEnterTS, const char *pszFunction, const char *pszFormat, ...)
{
    unsigned    uTS = getTimestamp();
    char       *pszMsg;
    int         cch;
    va_list     args;
    ULONG       cb;

    if ((int)hLog < 0 && openLog())
        return;

    pszMsg = alloca(CCHTMPMSGBUFFER);
    if (!pszMsg)
        return;

    cch = snprintf(pszMsg, CCHTMPMSGBUFFER, "%08x %02x Leav %s (%2d ms): ", uTS, getTid(), pszFunction, uTS - uEnterTS);
    va_start(args, pszFormat);
    cch += vsnprintf(pszMsg + cch, CCHTMPMSGBUFFER - cch, pszFormat, args);
    va_end(args);
    if (cch >= CCHTMPMSGBUFFER)
        memcpy(pszMsg + CCHTMPMSGBUFFER - 5, "...\n", 5);
    else if (pszMsg[cch - 1] != '\n')
        memcpy(&pszMsg[cch++], "\n", 2);
    DosWrite(hLog, pszMsg, cch, &cb);
}

/**
 * Output a log message.
 * A log message is considered to be one line and is appended a newline if
 * none was given.
 *
 * @param   uEnterTS        The timestamp returned by LogEnter.
 * @param   pszFunction     Name of the function which was entered.
 * @param   pszFormat       Format string for the message to log.
 * @param   ...             Arguments to the format string.
 */
void     __libc_LogMsg(unsigned uEnterTS, const char *pszFunction, const char *pszFormat, ...)
{
    unsigned    uTS = getTimestamp();
    char       *pszMsg;
    int         cch;
    ULONG       cb;
    va_list     args;

    if ((int)hLog < 0 && openLog())
        return;

    pszMsg = alloca(CCHTMPMSGBUFFER);
    if (!pszMsg)
        return;

    if (uEnterTS != ~0)
        cch = snprintf(pszMsg, CCHTMPMSGBUFFER, "%08x %02x Msg  %s (%2d ms): ", uTS, getTid(), pszFunction, uTS - uEnterTS);
    else
        cch = snprintf(pszMsg, CCHTMPMSGBUFFER, "%08x %02x Msg  %s: ", uTS, getTid(), pszFunction);
    va_start(args, pszFormat);
    cch += vsnprintf(pszMsg + cch, CCHTMPMSGBUFFER - cch, pszFormat, args);
    va_end(args);
    if (cch >= CCHTMPMSGBUFFER)
        memcpy(pszMsg + CCHTMPMSGBUFFER - 5, "...\n", 5);
    else if (pszMsg[cch - 1] != '\n')
        memcpy(&pszMsg[cch++], "\n", 2);
    DosWrite(hLog, pszMsg, cch, &cb);
}

