/* $Id: libclog.h 1157 2004-02-04 20:55:41Z bird $ */
/** @file
 *
 * Debug Logging Features - internal libc use only.
 *
 * InnoTek Systemberatung GmbH confidential
 *
 * Copyright (c) 2004 InnoTek Systemberatung GmbH
 * Author: knut st. osmundsen <bird-srcspam@anduin.net>
 *
 * All Rights Reserved
 *
 */

#ifndef __LIBCLOG_H__
#define __LIBCLOG_H__


#ifdef DEBUG_LOGGING
/*
 * Logging is enabled.
 */
/**
 * Output an enter function log message.
 * An enter message is considered to be one line and is appended a newline if
 * none was given.
 *
 * @returns Current timestamp.
 * @param   pszFunction     Name of the function which was entered.
 * @param   pszFormat       Format string to display arguments.
 * @param   ...             Arguments to the format string.
 */
extern unsigned __libc_LogEnter(const char *pszFunction, const char *pszFormat, ...);
/**
 * Output a leave function log message.
 * A leave message is considered to be one line and is appended a newline if
 * none was given.
 *
 * @param   uEnterTS        The timestamp returned by LogEnter.
 * @param   pszFunction     Name of the function which was entered.
 * @param   pszFormat       Format string to display the result.
 * @param   ...             Arguments to the format string.
 */
extern void     __libc_LogLeave(unsigned uEnterTS, const char *pszFunction, const char *pszFormat, ...);

/**
 * Output a log message.
 * A log message is considered to be one line and is appended a newline if
 * none was given.
 *
 * @param   uEnterTS        The timestamp returned by LogEnter.
 * @param   pszFunction     Name of the function which was entered.
 * @param   pszFormat       Format string for the message to log.
 * @param   ...             Arguments to the format string.
 */
extern void     __libc_LogMsg(unsigned uEnterTS, const char *pszFunction, const char *pszFormat, ...);

/** Macro to log a function entry. */
#define LIBCLOG_ENTER(...) \
    unsigned __libclog_uEnterTS__ = __libc_LogEnter(__PRETTY_FUNCTION__, __VA_ARGS__)
#define LIBCLOG_LEAVE(...) \
    __libc_LogLeave(__libclog_uEnterTS__, __PRETTY_FUNCTION__, __VA_ARGS__)
#define LIBCLOG_MSG(...) \
    __libc_LogLeave(__libclog_uEnterTS__, __PRETTY_FUNCTION__, __VA_ARGS__)

#else
/*
 * Logging is disabled.
 */
#define LIBCLOG_ENTER(...) //hmm
#define LIBCLOG_LEAVE(...)  do {} while (0)
#define LIBCLOG_MSG(...)    do {} while (0)

#endif

/* helpers */
#define LIBCLOG_RETURN_VOID()    do { LIBCLOG_LEAVE("ret void\n");                      return;      } while (0)
#define LIBCLOG_RETURN_INT(rc)   do { LIBCLOG_LEAVE("ret %d (%#x)\n", (rc), (rc));      return (rc); } while (0)
#define LIBCLOG_RETURN_UINT(rc)  do { LIBCLOG_LEAVE("ret %u (%#x)\n", (rc), (rc));      return (rc); } while (0)
#define LIBCLOG_RETURN_LONG(rc)  do { LIBCLOG_LEAVE("ret %ld (%#lx)\n", (rc), (rc));    return (rc); } while (0)
#define LIBCLOG_RETURN_ULONG(rc) do { LIBCLOG_LEAVE("ret %lu (%#lx)\n", (rc), (rc));    return (rc); } while (0)
#define LIBCLOG_RETURN_P(rc)     do { LIBCLOG_LEAVE("ret %p\n", (rc));                  return (rc); } while (0)

#endif
