/* $Id: filehandles.c 1131 2004-01-31 23:53:27Z bird $ */
/** @file
 *
 * LIBC File Handles.
 *
 * Copyright (c) 2003 knut st. osmundsen <bird-srcspam@anduin.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with This program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#define INCL_BASE
#define INCL_ERRORS
#define INCL_FSMACROS
#include <os2.h>
#include "libc-alias.h"
#include <malloc.h>
#include <sys/fcntl.h>
#include <errno.h>
#include <emx/io.h>
#include <emx/umalloc.h>
#include "syscalls.h"


/*******************************************************************************
*   Global Variables                                                           *
*******************************************************************************/
/** Mutex semaphor protecting all the global data.
 * @todo This should've been a read/write lock (multiple read, exclusive write).
 */
static _fmutex              gmtx;
/** Array of file handle pointers. */
static PLIBCFH             *gpapFHs;
/** Number of entires in the file handle table. */
static unsigned             gcFHs;

/** Array of preallocated handles - normal files only! */
static LIBCFH               gaPreAllocated[40];
/** Number of free handles in the preallocated array. */
static unsigned             gcPreAllocatedAvail;

extern int _fmode_bin;


/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int __libc_fhMoreHandles(unsigned cMin);
static int __libc_fhAllocate(int fh, unsigned fFlags, int cb, PLIBCFHOPS pOps, PLIBCFH *ppFH, int *pfh, int fOwnSem);


/**
 * Initiates the filehandle structures.
 *
 * @returns 0 on success.
 * @returns -1 on failure.
 */
int     _sys_init_filehandles(void)
{
    ULONG   cMaxFHs = 0;
    LONG    lDeltaFHs = 0;
    int     rc;
    int     i;

    /*
     * Only once.
     */
    if (gcFHs)
    {
#ifdef DEBUG
        __asm__("int $3");
#endif
        return 0;
    }

    /*
     * Init the mutex which should've been a read write lock.
     */
    rc = _fmutex_create(&gmtx, 0);
    if (rc)
        return rc;

    /*
     * Figure out the size of the pointer array and initiatlize it.
     */
    rc = DosSetRelMaxFH(&lDeltaFHs, &cMaxFHs);
    if (rc)
    {
#ifdef DEBUG
        __asm__("int $3");
#endif
        cMaxFHs = 20;
    }
    gcFHs = (cMaxFHs + 3) & ~3;     /* round up to nearest 4. */
    gpapFHs = _hcalloc(gcFHs, sizeof(gpapFHs[0]));
    if (!gpapFHs)
        return -1;

    gcPreAllocatedAvail = sizeof(gaPreAllocated) / sizeof(gaPreAllocated[0]);


    /*
     * Work thru the handles checking if they are in use.
     * Was previously done in init_files() in startup.c.
     *      We need to allocate the handles and initialize them accordingly.
     * Note! This duplicates a bit of the __ioctl2 functionality for
     *       reasons of speed.
     */
    for (i = 0; i < cMaxFHs; i++)
    {
        ULONG       fulType;
        ULONG       fulDevFlags;
        ULONG       fulMode;
        unsigned    fLibc;

        /*
         * Is it in use and if so what kind of handle?
         */
        if (    (rc = DosQueryHType((HFILE)i, &fulType, &fulDevFlags)) != NO_ERROR
            ||  (rc = DosQueryFHState((HFILE)i, &fulMode)) != NO_ERROR)
        {
#ifdef DEBUG
            if (rc != ERROR_INVALID_HANDLE)
                __asm__("int $3");
#endif
            continue;
        }

        /*
         * Determin initial flags.
         */
        switch (fulType & 0xff)
        {
            default: /* paranoia */
            case HANDTYPE_FILE:
                fLibc = F_FILE;
                break;
            case HANDTYPE_DEVICE:
                fLibc = F_DEV;
                /* @todo inherit O_NDELAY */
                break;
            case HANDTYPE_PIPE:
                fLibc = F_PIPE;
                break;
        }

        /*
         * There is some init_streams() dependency on all files to have non
         * zero flags. Sure, we might change this, but for now keep this in mind
         * when omitting O_TEXT.
         * @todo inherit non text files.
         */
        fLibc |= O_TEXT;

        /*
         * Read write flags.
         */
        switch (fulMode & (OPEN_ACCESS_READONLY | OPEN_ACCESS_WRITEONLY | OPEN_ACCESS_READWRITE))
        {
            case OPEN_ACCESS_READONLY:      fLibc |= O_RDONLY; break;
            case OPEN_ACCESS_WRITEONLY:     fLibc |= O_WRONLY; break;
            default: /* paranoia */
            case OPEN_ACCESS_READWRITE:     fLibc |= O_RDWR; break;
        }

        /*
         * Enforce rules on the three standard handles - is this strictly
         * speaking such a good idea? It's apparently required for the con device
         * as that is opened as readwrite for all three in OS/2 (to safe SFNs?).
         * However I think when it's not the console device this might be sort
         * of rushing to conclusions...
         */
        if (i < 2 && (fulDevFlags & 0x3 /* CON or KBD dev */))
        {
            if (i == 0)
                fLibc = (fLibc & ~O_ACCMODE) | O_RDONLY;
            else
                fLibc = (fLibc & ~O_ACCMODE) | O_WRONLY;
        }

        /*
         * Allocate handle and initialize handle.
         */
        rc = __libc_fhAllocate(i, fLibc, sizeof(LIBCFH), NULL, NULL, NULL, 1);
        if (rc)
        {
#ifdef DEBUG
            __asm__("int $3");
#endif
            return -1;
        }
    } /* fh init loop */

    return 0;
}


/**
 * Terminate the file handles.
 *
 * @returns 0 on success
 * @returns -1 on failure (what ever that means).
 */
int     _sys_term_filehandles(void)
{
    if (!gcFHs)
    {
#ifdef DEBUG
            __asm__("int $3");
#endif
        return 0;
    }
    gcFHs = 0;

    /*
     * Don't take the mutex as we might be in an crashing/hung process.
     * no need to make matters worse by potentially waiting for ever here.
     */
    _fmutex_close(&gmtx);
    memset(&gmtx, 0, sizeof(gmtx));

    free(gpapFHs);
    gpapFHs = NULL;

    return 0;
}


/**
 * Opens the dummy device for non OS/2 file handles.
 *
 * @returns 0 on success.
 * @returns OS/2 error on failure.
 * @param   fh  Filehandle to open the dummy device for.
 *              Use -1 if no special handle is requested.
 * @param   pfh Where to store the opened fake handle.
 */
static int __libc_fhOpenDummy(int fh, int *pfh)
{
    int     rc;
    ULONG   ulAction;
    HFILE   hFile = fh;
    FS_VAR();

    FS_SAVE_LOAD();
    rc = DosOpen("\\DEV\\NUL", &hFile, &ulAction, 0, FILE_NORMAL,
                 OPEN_ACTION_FAIL_IF_NEW | OPEN_ACTION_OPEN_IF_EXISTS,
                 OPEN_SHARE_DENYNONE | OPEN_ACCESS_WRITEONLY,
                 NULL);
    if (!rc)
    {
        if (fh != -1 && hFile != fh)
        {
            HFILE hDup = fh;
            rc = DosDupHandle(hFile, &hDup);
            if (!rc)
                *pfh = hDup;
            DosClose(hFile);
        }
        else
            *pfh = hFile;
    }
    FS_RESTORE();
    return rc;
}

/**
 * Frees handle data.
 *
 * @param   pFH Pointer to the handle to be freed.
 * @param   fh  Handle number which the data used to be associated with.
 *              Use -1 if it doesn't apply.
 * @remark  Must own the mutex upon entry!
 */
static void __libc_fhFreeHandle(PLIBCFH pFH)
{
    /*
     * Preallocated or heap?
     */
    if (    pFH >= &gaPreAllocated[0]
        &&  pFH <  &gaPreAllocated[sizeof(gaPreAllocated) / sizeof(gaPreAllocated[0])])
    {
        pFH->fFlags     = 0;
        pFH->iLookAhead = 0;
        pFH->pOps       = NULL;
        gcPreAllocatedAvail++;
    }
    else
        free(pFH);
}


/**
 * Try make room for more handles.
 *
 * @returns 0 on success.
 * @returns OS/2 error code on failure.
 * @param   cMin    Minimum number for the new max handle count.
 * @remark  Lock must be owned upon entry!
 */
static int __libc_fhMoreHandles(unsigned cMin)
{
    int     rc;
    ULONG   cNewMaxFHs = gcFHs;
    LONG    lDeltaFHs;
    FS_VAR();

    /*
     * How many handles?
     * Now, we must be somewhat sensibel about incrementing this number
     * into unreasonable values. If an application requires 10000+ files
     * it must take steps to tell that to OS/2. If he has we'll receive
     * cMin > 10000, else keep 10000 as a max limit for auto increase.
     */
    lDeltaFHs = 0;
    cNewMaxFHs = 0;
    FS_SAVE_LOAD();
    if (    DosSetRelMaxFH(&lDeltaFHs, &cNewMaxFHs)
        ||  cNewMaxFHs < gcFHs)
        cNewMaxFHs = gcFHs;

    if (cMin < cNewMaxFHs)
    {   /* auto increment. */
        cMin = cNewMaxFHs + 100;
        if (cMin > 10000 && gcFHs > 10000)
        {
            FS_RESTORE();
            return ERROR_TOO_MANY_OPEN_FILES;
        }
    }
    lDeltaFHs = cMin - cNewMaxFHs;

    /*
     * Now to the actual increment.
     * cMax will at this point hold the disired new max file handle count.
     * Only we must take precautions in case someone have secretly done
     * this without our knowledge.
     */
    rc = DosSetRelMaxFH(&lDeltaFHs, &cNewMaxFHs);
    if (!rc && cMin > cNewMaxFHs)
        rc = DosSetMaxFH((cNewMaxFHs = cMin));
    FS_RESTORE();
    if (!rc)
    {

        /*
         * Reallocate the array of handle pointers.
         */
        PLIBCFH * papNewFHs = _hrealloc(gpapFHs, cNewMaxFHs * sizeof(gpapFHs[0]));
        if (papNewFHs)
        {
            gpapFHs = papNewFHs;
            memset(&gpapFHs[gcFHs], 0, (cNewMaxFHs - gcFHs) * sizeof(gpapFHs[0]));
            gcFHs = cNewMaxFHs;
        }
    }

    return rc;
}


/**
 * Ensures space for the given handle.
 *
 * @returns 0 on success.
 * @returns OS/2 error code on failure.
 * @param   fh      Filehandle which must fit.
 */
int __libc_FHEnsureHandles(int fh)
{
    int rc;
    if (fh < (int)gcFHs)
    {
        /*
         * This holds as long as no fool have been changing the max FH
         * outside our control. And if someone does, they have themselves
         * to blame for any traps in DosDupHandle().
         */
        return 0;
    }

    if (_fmutex_request(&gmtx, 0))
        return -1;

    rc = __libc_fhMoreHandles(fh + 1);

    _fmutex_release(&gmtx);
    return rc;
}





/**
 * Allocates a file handle.
 *
 * @returns 0 on success.
 * @returns OS/2 error code and errno set to the corresponding error number.
 * @param   fh      Number of the filehandle to allocate.
 *                  Will fail if the handle is in use.
 *                  Use -1 for any handle.
 * @param   fFlags  Initial flags.
 * @param   cb      Size of the file handle.
 *                  Must not be less than the mandatory size (sizeof(LIBCFH)).
 * @param   pOps    Value of the pOps field, NULL not allowed.
 * @param   ppFH    Where to store the allocated file handle struct pointer. (NULL allowed)
 * @param   pfh     Where to store the number of the filehandle allocated.
 * @param   fOwnSem Set if we should not take or release the semaphore.
 * @remark  The preallocated handles make this function somewhat big and messy.
 */
static int __libc_fhAllocate(int fh, unsigned fFlags, int cb, PLIBCFHOPS pOps, PLIBCFH *ppFH, int *pfh, int fOwnSem)
{
    PLIBCFH     pFH;
    int         rc;
    FS_VAR();

    /*
     * If we know we cannot use the preallocated handles it's prefered to
     * do the allocation outside the mutex.
     */
    pFH = NULL;
    if (cb > sizeof(LIBCFH) || !gcPreAllocatedAvail)
    {
        pFH = _hmalloc(cb);
        if (!pFH)
            return ERROR_NOT_ENOUGH_MEMORY;
    }

    /*
     * Now take the lock.
     */
    if (!fOwnSem && _fmutex_request(&gmtx, 0))
        return -1;

    /*
     * Now is the time to try use the pre allocated handles.
     */
    if (!pFH)
    {
        if (cb == sizeof(LIBCFH) && gcPreAllocatedAvail)
        {
            PLIBCFH pFHSearch;
            for (pFHSearch = &gaPreAllocated[0];
                 pFH < &gaPreAllocated[sizeof(gaPreAllocated) / sizeof(gaPreAllocated[0])];
                 pFHSearch++)
                if (!pFHSearch->fFlags)
                {
                    gcPreAllocatedAvail--;
                    pFH = pFHSearch;
                    break;
                }
#ifdef DEBUG
            if (!pFH) __asm__("int $3");
#endif
        }

        /*
         * If the pre allocated handle table for some reason was
         * full, we'll allocate the handle here.
         */
        if (!pFH)
            pFH = _hmalloc(cb);
    }

    if (pFH)
    {
        /*
         * Initiate the handle data.
         */
        pFH->fFlags     = fFlags;
        pFH->iLookAhead = -1;
        pFH->pOps       = pOps;

        /*
         * Insert into the handle table.
         */
        if (fh == -1)
        {
            /*
             * Non OS/2 filehandle.
             * Open dummy device to find a handle number.
             */
#ifdef DEBUG
            if (!pOps)
                __asm__("int $3");
#endif
            rc = __libc_fhOpenDummy(-1, &fh);
            if (!rc)
            {
                /*
                 * Enough space in the handle table?
                 */
                if (fh >= gcFHs)
                    rc = __libc_fhMoreHandles(fh + 1);
                if (!rc)
                {
                    /*
                     * Free any old filehandle which death we didn't catch
                     * and insert the new one.
                     */
                    if (gpapFHs[fh])
                        __libc_fhFreeHandle(gpapFHs[fh]);
                    gpapFHs[fh] = pFH;
                }
                else
                {
                    /* failure: close dummy */
                    FS_SAVE_LOAD();
                    DosClose((HFILE)fh);
                    FS_RESTORE();
                    fh = -1;
                }
            }
        }
        else
        {
            /*
             * Special handle.
             * Make sure there are enough file handles available.
             */
            rc = 0;
            if (fh >= gcFHs)
                rc = __libc_fhMoreHandles(fh + 1);

            if (!rc)
            {
                /*
                 * If not OS/2 filehandle, we have to make a fake handle.
                 */
                if (pOps)
                    rc = __libc_fhOpenDummy(-1, &fh);

                if (!rc)
                {
                    /*
                     * Free any old filehandle which death we didn't catch
                     * and insert the new one.
                     */
                    if (gpapFHs[fh])
                        __libc_fhFreeHandle(gpapFHs[fh]);
                    gpapFHs[fh] = pFH;
                }
            }
        }
    }
    else
        rc = ERROR_NOT_ENOUGH_MEMORY;

    /*
     * On failure cleanup any mess!
     */
    if (rc)
    {
        _sys_set_errno(rc);
        if (pFH)
            __libc_fhFreeHandle(pFH);
        pFH = NULL;
        fh = -1;
    }

    /*
     * Return.
     */
    if (ppFH)
        *ppFH = pFH;
    if (pfh)
        *pfh = fh;

    if (!fOwnSem)
        _fmutex_release(&gmtx);
    return rc;
}

/**
 * Allocates a file handle.
 *
 * @returns 0 on success.
 * @returns OS/2 error code and errno set to the corresponding error number.
 * @param   fh      Number of the filehandle to allocate.
 *                  Will fail if the handle is in use.
 *                  Use -1 for any handle.
 * @param   fFlags  Initial flags.
 * @param   cb      Size of the file handle.
 *                  Must not be less than the mandatory size (sizeof(LIBCFH)).
 * @param   pOps    Value of the pOps field, NULL not allowed.
 * @param   ppFH    Where to store the allocated file handle struct pointer. (NULL allowed)
 * @param   pfh     Where to store the number of the filehandle allocated.
 * @remark  The preallocated handles make this function somewhat big and messy.
 */
int __libc_FHAllocate(int fh, unsigned fFlags, int cb, PLIBCFHOPS pOps, PLIBCFH *ppFH, int *pfh)
{
    return __libc_fhAllocate(fh, fFlags, cb, pOps, ppFH, pfh, 0);
}


/**
 * Close (i.e. free) a file handle.
 *
 * @returns 0 on success.
 * @returns OS/2 error code on failure and errno set to corresponding error number.
 * @param   fh      Filehandle to close.
 */
int __libc_FHClose(int fh)
{
    PLIBCFH     pFH;
    ULONG       rc;
    FS_VAR();

    if (_fmutex_request(&gmtx, 0))
        return -1;

    /*
     * Validate input.
     */
    if (    (fh < 0 && fh >= gcFHs)
        ||  !gpapFHs[fh])
    {
        _fmutex_release(&gmtx);
#ifdef DEBUG
        __asm__("int $3");
#endif
        errno = EBADF;
        return -1;
    }

    pFH = gpapFHs[fh];

    /*
     * If this is an non OS/2 handle, let the subsystem clean it up first.
     */
    rc = 0;
    if (pFH->pOps && pFH->pOps->pfnClose)
        rc = pFH->pOps->pfnClose(pFH, fh);

    /*
     * We should continue closing the OS/2 handle if the previuos step
     * were successful or if the handle has become invalid.
     */
    if (!rc || rc == ERROR_INVALID_HANDLE || rc == -EBADF)
    {
        int rc2;
        /*
         * Close the OS/2 handle and remove the handle from the array
         * making the space available to others.
         * - This ain't the way which scales best, but it's the safe way.
         */
        FS_SAVE_LOAD();
        rc2 = DosClose(fh);
        FS_RESTORE();
        if (!pFH->pOps)
            rc = rc2;
#ifdef DEBUG
        else if (rc2) __asm__("int $3");
#endif

        /*
         * If we successfully freed the handle, or it had become invalid
         * we will free it now.
         */
        if (!rc || rc == ERROR_INVALID_HANDLE)
        {
            gpapFHs[fh] = NULL;
            __libc_fhFreeHandle(pFH);
        }
    }

    _fmutex_release(&gmtx);

    if (rc)
    {
        if (rc > 0)
            _sys_set_errno(rc);
        else
            errno = rc;
    }

    return rc;
}


/**
 * Get the LIBC handle structure corresponding to a filehandle.
 *
 * @returns Pointer to handle structure on success.
 * @returns NULL on failure.
 * @param   fh  Handle to lookup.
 */
PLIBCFH __libc_FH(int fh)
{
    PLIBCFH pFH = NULL;

    /** @todo shared access */
    if (!_fmutex_request(&gmtx, 0))
    {
        if (fh >= 0 && fh < gcFHs)
        {
            pFH = gpapFHs[fh];
            if (!pFH)
            {
                /*
                 * Try import the handle.
                 */
                ULONG       rc;
                ULONG       fulType;
                ULONG       fulDevFlags;
                ULONG       fulMode;
                unsigned    fLibc;

                /*
                 * Is it in use and if so what kind of handle?
                 */
                if (    (rc = DosQueryHType((HFILE)fh, &fulType, &fulDevFlags)) != NO_ERROR
                    ||  (rc = DosQueryFHState((HFILE)fh, &fulMode)) != NO_ERROR)
                {
                    errno = EBADF;
                    _fmutex_release(&gmtx);
                    return NULL;
                }

                /*
                 * Determin initial flags.
                 */
                switch (fulType & 0xff)
                {
                    default: /* paranoia */
                    case HANDTYPE_FILE:
                        fLibc = F_FILE;
                        break;
                    case HANDTYPE_DEVICE:
                        fLibc = F_DEV;
                        /* @todo inherit O_NDELAY */
                        break;
                    case HANDTYPE_PIPE:
                        fLibc = F_PIPE;
                        break;
                }

                /*
                 * Read write flags.
                 */
                switch (fulMode & (OPEN_ACCESS_READONLY | OPEN_ACCESS_WRITEONLY | OPEN_ACCESS_READWRITE))
                {
                    case OPEN_ACCESS_READONLY:      fLibc |= O_RDONLY; break;
                    case OPEN_ACCESS_WRITEONLY:     fLibc |= O_WRONLY; break;
                    default: /* paranoia */
                    case OPEN_ACCESS_READWRITE:     fLibc |= O_RDWR; break;
                }

                /*
                 * Textflag.
                 */
                if (!_fmode_bin)
                    fLibc |= O_TEXT;


                /*
                 * Allocate a new handle for this filehandle.
                 */
                rc = __libc_fhAllocate(fh, fLibc, sizeof(LIBCFH), NULL, &pFH, NULL, 1);
                if (rc)
                    pFH = NULL;
            }
        }

        _fmutex_release(&gmtx);
    }

    return pFH;
}

