/******************************************************************************
 * apm.c - Functions to interface with the APM driver.
 *
 * Copyright (c) 2011 thi.guten Software Development
 * Copyright (c) 2011 Mensys B.V.
 *
 * Authors: Christian Mueller, Markus Thielen
 *
 * Parts copied from/inspired by the Linux AHCI driver;
 * those parts are (c) Linux AHCI/ATA maintainers
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "os2ahci.h"

#include <apmcalls.h>

/* -------------------------- macros and constants ------------------------- */

/* ------------------------ typedefs and structures ------------------------ */

/* -------------------------- function prototypes -------------------------- */

/* ------------------------ global/static variables ------------------------ */

USHORT _far _cdecl apm_event    (APMEVENT _far *evt);

/* ----------------------------- start of code ----------------------------- */

/******************************************************************************
 * Connect to APM driver and register for power state change events.
 */
void apm_init(void)
{
  USHORT rc;

  /* connect to APM driver */
  if ((rc = APMAttach()) != 0) {
    dprintf("couldn't connect to APM driver (rc = %d)\n", rc);
    return;
  }

  /* register for suspend/resume events */
  if ((rc = APMRegister(apm_event, APM_NOTIFYSETPWR |
                                   APM_NOTIFYNORMRESUME |
                                   APM_NOTIFYCRITRESUME, 0)) != 0) {
    dprintf("couldn't register for power event notificatins (rc = %d)\n", rc);
    return;
  }
}

/******************************************************************************
 * APM event handler
 */
USHORT _far _cdecl apm_event(APMEVENT _far *evt)
{
  USHORT msg = (USHORT) evt->ulParm1;

  dprintf("received APM event: 0x%lx/0x%lx\n");

  switch (msg) {

  case APM_SETPWRSTATE:
    if (evt->ulParm2 >> 16 != APM_PWRSTATEREADY) {
      /* we're suspending */
      apm_suspend();
    }
    break;

  case APM_NORMRESUMEEVENT:
  case APM_CRITRESUMEEVENT:
    /* we're resuming */
    apm_resume();
    break;

  default:
    dprintf("unknown APM event; ignoring...\n");
    break;
  }

  return(0);
}

/******************************************************************************
 * APM suspend handler. In a nutshell, it'll turn of interrupts and flush all
 * write caches.
 */
void apm_suspend(void)
{
  int a;
  int p;
  int d;

  dprintf("apm_suspend()\n");

  /* restart all ports with interrupts disabled */
  for (a = 0; a < ad_info_cnt; a++) {
    AD_INFO *ai = ad_infos + a;

    lock_adapter(ai);
    for (p = 0; p <= ai->port_max; p++) {
      /* wait until all active commands have completed on this port */
      while (ahci_port_busy(ai, p)) {
        msleep(250);
      }

      /* restart port with interrupts disabled */
      ahci_stop_port(ai, p);
      ahci_start_port(ai, p, 0);

      /* flush cache on all attached devices */
      for (d = 0; d <= ai->ports[p].dev_max; d++) {
        if (ai->ports[p].devs[d].present) {
	  ahci_flush_cache(ai, p, d);
	}
      }
    }
  }

  /* reset init_complete so that we can process IORBs without interrupts */
  init_complete = 0;

  /* restore BIOS configuration for each adapter and release the adapter */
  for (a = 0; a < ad_info_cnt; a++) {
    ahci_restore_bios_config(ad_infos + a);
    unlock_adapter(ad_infos + a);
  }

  dprintf("apm_suspend() finished\n");
}

/******************************************************************************
 * APM resume handler. All ports are restarted with interrupts enabled using
 * the same function as the IOCM_COMPLETE_INIT handler does.
 */
void apm_resume(void)
{
  int a;

  dprintf("apm_resume()\n");

  for (a = 0; a < ad_info_cnt; a++) {
    AD_INFO *ai = ad_infos + a;

    /* Complete initialization of this adapter; this will restart the ports
     * with interrupts enabled and take care of whatever else needs to be
     * done to get the adapter and its ports up and running.
     */
    lock_adapter(ai);
    ahci_complete_init(ai);
  }

  /* tell the driver we're again fully operational */
  init_complete = 1;

  /* unlock all adapters now that we have set the init_complete flag */
  for (a = 0; a < ad_info_cnt; a++) {
    unlock_adapter(ad_infos + a);
  }

  dprintf("apm_resume() finished\n");
}
