/******************************************************************************
 * atapi.c - ATAPI command processing
 *
 * Copyright (c) 2010 Christian Mueller, Markus Thielen. 
 * Parts copied from/inspired by the Linux AHCI driver; 
 * those parts are (c) Linux AHCI/ATA maintainers
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "os2ahci.h"
#include "ata.h"
#include "atapi.h"

/* -------------------------- macros and constants ------------------------- */

/* ------------------------ typedefs and structures ------------------------ */

/* -------------------------- function prototypes -------------------------- */

static void atapi_req_sense_pp      (IORBH _far *iorb);


/* ------------------------ global/static variables ------------------------ */

/* ----------------------------- start of code ----------------------------- */

/******************************************************************************
 * Get device or media geometry. This function is not expected to be called.
 */
int atapi_get_geometry(IORBH _far *iorb, int slot)
{
  iorb_seterr(iorb, IOERR_CMD_NOT_SUPPORTED);
  return(-1);
}

/******************************************************************************
 * Test whether unit is ready. This function is not expected to be called.
 */
int atapi_unit_ready(IORBH _far *iorb, int slot)
{
  dprintf("atapi_unit_ready called\n");
  iorb_seterr(iorb, IOERR_CMD_NOT_SUPPORTED);
  return(-1);
}

/******************************************************************************
 * Read sectors from AHCI device.
 */
int atapi_read(IORBH _far *iorb, int slot)
{
  IORB_EXECUTEIO _far *io = (IORB_EXECUTEIO _far *) iorb;
  ATAPI_CDB_12 cdb;
  AD_INFO *ai = ad_infos + iorb_unit_adapter(iorb);
  USHORT count = io->BlockCount - io->BlocksXferred;
  USHORT sg_indx;
  USHORT sg_cnt;
  int p = iorb_unit_port(iorb);
  int d = iorb_unit_device(iorb);
  int rc;

  /* translate read command to SCSI/ATAPI READ12 command.
   * READ12 seems to be the most supported READ variant - according to MMC,
   * and its enough even for BluRay.
   */
  memset(&cdb, 0x00, sizeof(cdb));
  cdb.cmd = ATAPI_CMD_READ_12;
  SET_CDB_32(cdb.lba, io->RBA + io->BlocksXferred);

  do {
    /* update sector count (might have been updated due to S/G limitations) */
    SET_CDB_32(cdb.trans_len, (u32) count);

    /* update S/G count and index */
    sg_indx = ata_get_sg_indx(io);
    sg_cnt = io->cSGList - sg_indx;

    /* issue command */
    rc = ata_cmd(ai, p, d, slot, ATA_CMD_PACKET,
                 AP_ATAPI_CMD, (void _far *) &cdb, sizeof(cdb),
                 AP_SGLIST,    io->pSGList + sg_indx, (u16) sg_cnt,
                 AP_DEVICE,    0x4000,
                 AP_END);

    if (rc > 0) {
      /* couldn't map all S/G elements */
      ata_max_sg_cnt(io, sg_indx, (USHORT) rc, &sg_cnt, &count);
    }
  } while (rc > 0 && sg_cnt > 0);
    
  if (rc == 0) {
    add_workspace(iorb)->blocks = count;
    add_workspace(iorb)->ppfunc = ata_read_pp;

  } else if (rc > 0) {
    iorb_seterr(iorb, IOERR_CMD_SGLIST_BAD);

  } else {
    iorb_seterr(iorb, IOERR_CMD_ADD_SOFTWARE_FAILURE);
  }

  return(rc);
}

/******************************************************************************
 * Verify readability of sectors on AHCI device. This function is not expected
 * to be called.
 */
int atapi_verify(IORBH _far *iorb, int slot)
{
  dprintf("atapi_verify called\n");
  iorb_seterr(iorb, IOERR_CMD_NOT_SUPPORTED);
  return(-1);
}

/******************************************************************************
 * Write sectors to AHCI device. This function is not expected to be called.
 */
int atapi_write(IORBH _far *iorb, int slot)
{
  dprintf("atapi_write called\n");
  iorb_seterr(iorb, IOERR_CMD_NOT_SUPPORTED);
  return(-1);
}

/******************************************************************************
 * Execute ATAPI command.
 */
int atapi_execute_cdb(IORBH _far *iorb, int slot)
{
  IORB_ADAPTER_PASSTHRU _far *pt = (IORB_ADAPTER_PASSTHRU _far *) iorb;
  int rc;

  /* we do not perform the S/G limitation recovery loop here:
   * "ADDs are not required to iterate commands through the CDB PassThru
   * mechanism:" -- Storage Device Driver Reference, Scatter/Gather Lists
   */
  rc = ata_cmd(ad_infos + iorb_unit_adapter(iorb),
               iorb_unit_port(iorb),
               iorb_unit_device(iorb),
               slot, ATA_CMD_PACKET,
               AP_ATAPI_CMD, (void _far *) pt->pControllerCmd, 
                             pt->ControllerCmdLen,
               AP_SGLIST,    pt->pSGList, pt->cSGList,
               AP_END);

  if (rc == 0) {
    add_workspace(iorb)->blocks = pt->cSGList;
    /* no pp func needed */

  } else if (rc > 0) {
    iorb_seterr(iorb, IOERR_CMD_SGLIST_BAD);

  } else {
    iorb_seterr(iorb, IOERR_CMD_ADD_SOFTWARE_FAILURE);
  }
  
  return(rc);
}


/******************************************************************************
 * Request sense information for a failed command.
 *
 * NOTE: This function must be called right after an ATAPI command has failed
 *       and before any other commands are queued on the corresponding device.
 *       This function is typically called in the port restart context hook
 *       which is triggered by an AHCI error interrupt.
 *
 */
int atapi_req_sense(IORBH _far *iorb, int slot)
{
  ADD_WORKSPACE _far *aws = add_workspace(iorb);
  ATAPI_CDB_REQ_SENSE cdb;
  int rc;
  
  /* allocate sense buffer in ADD workspace */
  aws->buf = malloc(ATAPI_SENSE_LEN);
  if (aws->buf == NULL) {
    iorb_seterr(iorb, IOERR_CMD_SW_RESOURCE);
    return(-1);
  }
  memset(aws->buf, 0x00, ATAPI_SENSE_LEN);

  /* prepare request sense command */
  memset(&cdb, 0x00, sizeof(cdb));
  cdb.cmd = ATAPI_CMD_REQUEST_SENSE;
  aws->ppfunc = atapi_req_sense_pp;
  rc = ata_cmd(ad_infos + iorb_unit_adapter(iorb),
               iorb_unit_port(iorb),
               iorb_unit_device(iorb),
               slot,
               ATA_CMD_PACKET,
               AP_ATAPI_CMD, (void _far*) &cdb, sizeof(cdb),
               AP_VADDR, (void _far *) aws->buf, ATAPI_SENSE_LEN,
               AP_END);

  if (rc > 0) {
    /* should never happen - we got 64 bytes here */
    iorb_seterr(iorb, IOERR_CMD_SGLIST_BAD);

  } else {
    /* we failed to get info about an error -> return
     * non specific device error
     */
    iorb_seterr(iorb, IOERR_DEVICE_NONSPECIFIC);
  }

  return(rc);
}

/******************************************************************************
 * Post processing function for ATAPI request sense; examines the sense
 * data returned and maps sense info to IORB error info.
 */
static void atapi_req_sense_pp(IORBH _far *iorb)
{
  ADD_WORKSPACE _far *aws = add_workspace(iorb);
  ATAPI_SENSE_DATA *psd = (ATAPI_SENSE_DATA *) aws->buf;

  /* map sense data to some IOERR_ value */
  switch (ATAPI_GET_SENSE(psd)) {

  case ASENSE_NO_SENSE:
  case ASENSE_RECOVERED_ERROR:
    /* no error */
    break;

  case ASENSE_NOT_READY:
    iorb_seterr(iorb, IOERR_UNIT_NOT_READY);
    break;

  case ASENSE_UNIT_ATTENTION:
    iorb_seterr(iorb, IOERR_MEDIA_CHANGED);
    break;

  case ASENSE_MEDIUM_ERROR:
    iorb_seterr(iorb, IOERR_MEDIA);
    break;

  case ASENSE_ILLEGAL_REQUEST:
    iorb_seterr(iorb, IOERR_CMD_SYNTAX);
    break;

  case ASENSE_DATA_PROTECT:
    iorb_seterr(iorb, IOERR_MEDIA_WRITE_PROTECT);
    break;

  case ASENSE_BLANK_CHECK:
    iorb_seterr(iorb, IOERR_MEDIA_NOT_FORMATTED);
    break;

  case ASENSE_ABORTED_COMMAND:
  case ASENSE_COPY_ABORTED:
    iorb_seterr(iorb, IOERR_CMD_ABORTED);
    break;
  
  default:
    iorb_seterr(iorb, IOERR_DEVICE_NONSPECIFIC);
    break;
  }

  /* free sense buffer */
  aws_free(aws);

}
