/******************************************************************************

  lbatest.c - simple (dumb) test program for os2ahci

  Writes the actual LBA to each sector on a hard disk. The purpose is to
  find out if the sector mapping of os2ahci.add is correct.

  To run the test, attach a HD to an AHCI controller with os2ahci.add loaded
  and run the test in write mode, then attach the disk to an IDE adapter
  with an S506 driver and run lbatest in read mode.

  Author: Markus Thielen

  Compilation (Watcom): wcl386 -bt=os2 lbatest.c

  Copyright (c) 2010 by thi.guten Software development, www.thiguten.de

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

******************************************************************************/

/*---- includes -------------------------------------------------------------*/

#define INCL_DOS
#define INCL_ERRORS
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#include <os2.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <signal.h>

#define SECTOR_SIZE        512
#define SECTORS_PER_WRITE  200


/*--- function prototypes ---------------------------------------------------*/

void          usage            (void);

int           write_test       (char *drv);

int           read_test        (char *drv);

void          signal_handler   (int sig_no);

/*--- global data -----------------------------------------------------------*/

unsigned long lba_pos;
unsigned long first_wrong_sector = 0xffffffff;
char mode;

/*--- start of code ---------------------------------------------------------*/

/******************************************************************************
 * main()
 */
int main(int argc, char **argv)
{
  char *drv;
  HFILE hf;

  if (argc < 3) {
    usage();
    return -1;
  }

  /* get arguments */
  drv = argv[1];
  mode = toupper(argv[2][0]);
  if (mode != 'W' && mode != 'R') {
    usage();
    return -1;
  }

  if (drv[1] != ':' && drv[2] != '\0') {
    /* we want a drive letter, which is a letter and a colon */
    usage();
    return -1;
  }

  if (mode == 'W') {
    return write_test(drv);
  }
  if (mode == 'R') {
    return read_test(drv);
  }

  return 0;
}

/******************************************************************************
 * write_test() - write each sector's address to the sector itself
 */
int write_test(char *drv)
{
  unsigned long lba;
  unsigned long ret;
  HFILE hf_disk;
  unsigned long action;
  unsigned long cb_written;
  int rc = 0;
  float gbf = 1024.0 * 1024.0 * 1024.0 / 512.0;
  unsigned long i;
  unsigned char *wbuf;
  char buf[100];

  /* ask for confirmation, we destroy the drive's contents */
  printf("I'm going to destroy ALL DATA on drive %s now.\n"
	 "type 'DESTROY' to continue, anything else to bail out.\n", drv);
  if (gets(buf) == NULL || strcmp(buf, "DESTROY")) {
    return -1;
  }

  /* open drive */
  ret = DosOpen(drv, &hf_disk, &action, 0, 0, OPEN_ACTION_OPEN_IF_EXISTS,
		OPEN_FLAGS_DASD | OPEN_FLAGS_FAIL_ON_ERROR |
		OPEN_FLAGS_WRITE_THROUGH | OPEN_FLAGS_NO_CACHE |
		OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_WRITEONLY,
		NULL);
  if (ret != NO_ERROR) {
    fprintf(stderr, "Failed to open disk %s for writing: %d\n", drv, ret);
    return -1;
  }

  /* lock disk */
  ret = DosDevIOCtl(hf_disk, IOCTL_DISK, DSK_LOCKDRIVE, NULL, 0,
		    &action, NULL, 0, &cb_written);
  if (ret != NO_ERROR) {
    fprintf(stderr, "Failed to lock drive, code %d\n", ret);
    rc = -1;
    goto cleanup;
  }

  /* allocate big write buffer */
  wbuf = calloc(SECTOR_SIZE, SECTORS_PER_WRITE);

  /* go... */
  for (lba = 0; ; lba += SECTORS_PER_WRITE) {

    /* prepare buffer; set the first 4 bytes of each sector
     * to its LBA */
      for (i = 0; i < SECTORS_PER_WRITE; i++) {
        *((unsigned long *)(wbuf + i * SECTOR_SIZE)) = lba + i;
      }

    /* write buffer to disk */
    ret = DosWrite(hf_disk, wbuf, SECTOR_SIZE * SECTORS_PER_WRITE, &cb_written);
    if (cb_written < SECTOR_SIZE * SECTORS_PER_WRITE) {
      /* done?! */
      goto cleanup;
    }
    if (ret != NO_ERROR) {
      fprintf(stderr, "\nFailed to write to disk %s, code %d\n", drv, ret);
      rc = -1;
      goto cleanup;
    }

    /* write progress */
    printf("\r%dk sectors written (%0.02f GB)", lba / 1000, (float)lba / gbf);
    
  }

cleanup:
  /* unlock drive */
  DosDevIOCtl(hf_disk, IOCTL_DISK, DSK_UNLOCKDRIVE, NULL, 0, &action,
              NULL, 0, &cb_written);
  DosClose(hf_disk);
  free(wbuf);
  return rc;

}

/******************************************************************************
 * read_test() - read each sector's first 4 bytes and compare to its LBA
 */
int read_test(char *drv)
{
  unsigned long ret;
  HFILE hf_disk;
  unsigned long action;
  unsigned long cb_read;
  char buf[SECTOR_SIZE];
  int rc = 0;
  unsigned long err_cnt = 0;
  float gbf = 1024.0 * 1024.0 * 1024.0 / 512.0;

  /* open drive */
  ret = DosOpen(drv, &hf_disk, &action, 0, 0, OPEN_ACTION_OPEN_IF_EXISTS,
		OPEN_FLAGS_DASD | OPEN_FLAGS_FAIL_ON_ERROR |
		OPEN_FLAGS_WRITE_THROUGH | OPEN_FLAGS_NO_CACHE |
		OPEN_FLAGS_SEQUENTIAL | OPEN_SHARE_DENYREADWRITE |
		OPEN_ACCESS_READONLY,
		NULL);
  if (ret != NO_ERROR) {
    fprintf(stderr, "Failed to open disk %s for reading: %d\n", drv, ret);
    return -1;
  }

  /* lock disk */
  ret = DosDevIOCtl(hf_disk, IOCTL_DISK, DSK_LOCKDRIVE, NULL, 0,
		    &action, NULL, 0, &cb_read);
  if (ret != NO_ERROR) {
    fprintf(stderr, "Failed to lock drive, code %d\n", ret);
    rc = -1;
    goto cleanup;
  }

  /* catch Ctrl+C */
  signal(SIGINT, signal_handler);

  /* go... */
  memset(buf, 0x00, sizeof(buf));
  for (lba_pos = 0; ; lba_pos++) {
    ret = DosRead(hf_disk, buf, SECTOR_SIZE, &cb_read);
    if (ret != NO_ERROR) {
      fprintf(stderr, "\nFailed to read from disk %s, code %d\n", drv, ret);
      rc = -1;
      goto cleanup;
    }

    if (*((unsigned long*) buf) != lba_pos) {
      printf("\nWrong sector number: read %d from sector %d\n",
	      *((unsigned long*) buf), lba_pos);
      err_cnt++;
      if (first_wrong_sector == 0xffffffff) {
        first_wrong_sector = lba_pos;
      }
    }

    /* progress */
    if (lba_pos % 100 == 0) {
      printf("\r%dk sectors read (%0.02f GB)", lba_pos / 1000, 
             (float) lba_pos / gbf);
    }
  }

cleanup:
  /* unlock drive */
  DosDevIOCtl(hf_disk, IOCTL_DISK, DSK_UNLOCKDRIVE, NULL, 0, &action,
              NULL, 0, &cb_read);
  DosClose(hf_disk);

  /* print summary */
  printf("Found %d logical errors\n", err_cnt);
  if (first_wrong_sector != 0xffffffff) {
    printf("First wrong sector was %u.\n", err_cnt, first_wrong_sector);
  }

  return rc;
}

/******************************************************************************
 * usage() - print usage summary to STDOUT
 */
void usage(void)
{
  printf("lbatest for os2ahci.add\n"
	 "Usage:\n\n"
	 "lbatest <drive letter> <mode>\n\n"
	 "where mode is either W for writing or R for reading.\n\n"
	 "In write mode, this program COMPLETELY DESTROYS ALL DATA on the specified\n"
	 "partition/disc. It writes the address of each sector to the sector itself.\n\n"
	 "In read mode, each sector is read and its address verified.\n");
}


/******************************************************************************
 * signal_handler for SIGINT - prints summary to STDOUT
 */
void signal_handler(int sig_no)
{

  if (sig_no == SIGINT && mode == 'R') {
    /* read mode interrupted; show summary */
    printf("\n\nLast block read:    %u\n", lba_pos);
    if (first_wrong_sector != 0xffffffff) {
      printf("First wrong sector: %u\n", first_wrong_sector);
    } else {
      printf("All sectors read ok\n");
    }
  }

  exit(0);

}
