/******************************************************************************

  readsect.c - simple (dumb) HD sector read program

  Author: Markus Thielen

  Compilation (Watcom): wcl386 -bt=os2 readsect.c

  Copyright (c) 2010 by thi.guten Software development, www.thiguten.de

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

******************************************************************************/

/*---- includes -------------------------------------------------------------*/

#define INCL_DOS
#define INCL_ERRORS
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#include <os2.h>
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

#define SECTOR_SIZE        512
#define SECTORS_PER_READ   4


/*--- function prototypes ---------------------------------------------------*/

void          usage            (void);

int           read_sectors     (char *drv, unsigned long sectors_to_read, 
                                FILE *fpo);

void          signal_handler   (int sig_no);

/*--- global data -----------------------------------------------------------*/

unsigned long sectors_per_read = SECTORS_PER_READ;


/*--- start of code ---------------------------------------------------------*/

/******************************************************************************
 * main()
 */
int main(int argc, char **argv)
{
  char drv[50];
  char *output;
  FILE *fpo; /* output file */
  int ret;
  unsigned long sectors_to_read;

  if (argc < 4) {
    usage();
    return -1;
  }

  if (isalpha(argv[1][0])) {
    sprintf(drv, "\\\\.\\%c", toupper(argv[1][0]));
  } else if (isdigit(argv[1][0])) {
    sprintf(drv, "\\\\.\\Physical_Disk%c", argv[1][0]);
  } else {
    usage();
    return -1;
  }

  /* get sectors to read */
  sectors_to_read = strtoul(argv[2], NULL, 10);

  /* get output file */
  output = argv[3];
  if (*output == '-' && output[1] == 0) {
    /* use stdout */
    fpo = stdout;
  } else {
    fpo = fopen(output, "wb");
    if (!fpo) {
      perror("Failed to open output file");
      return(-1);
    }
  }

  if (argc > 4) {
    sectors_per_read = (unsigned long) atoi(argv[4]);
    if (sectors_per_read == 0) {
      sectors_per_read = SECTORS_PER_READ;
    }
  }

  printf("\nreadsect started; using %d sectors per read\n", 
         (int) sectors_per_read);

  /* go */
  ret = read_sectors(drv, sectors_to_read, fpo);
  if (fpo != stdout) {
    fclose(fpo);
  }

  return ret;

}

/******************************************************************************
 * read_sectors() - read sectors and dump to output file
 */
int read_sectors(char *drv, unsigned long sectors_to_read, FILE *fpo)
{
  unsigned long ret;
  HFILE hf_disk;
  unsigned long action;
  unsigned long cb_read;
  char *buf;
  int rc = 0;
  unsigned long cb_take;
  unsigned long sectors_take;
  unsigned long sectors_read = 0;

  buf = calloc(SECTOR_SIZE, sectors_per_read);

  /* open drive */
  printf("Getting handle for drive %s\n", drv);
  ret = DosOpen(drv, &hf_disk, &action, 0, 0, 
                OPEN_ACTION_OPEN_IF_EXISTS,
                OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY,
                NULL);
  if (ret != NO_ERROR) {
    fprintf(stderr, "Failed to open disk %s for reading: %d\n", drv, ret);
    return -1;
  }

  /* go... */
  memset(buf, 0x00, sizeof(buf));
  while (sectors_read < sectors_to_read) {

    sectors_take = min(sectors_to_read - sectors_read, sectors_per_read);
    cb_take = SECTOR_SIZE * sectors_take;

    ret = DosRead(hf_disk, buf, cb_take, &cb_read);
    if (ret != NO_ERROR) {
      fprintf(stderr, "\nFailed to read from disk %s, code %d\n", drv, ret);
      rc = -1;
      break;
    }

    if (cb_read == 0) {
      break;
    }

    if (cb_read != cb_take) {
      printf("\n\nRead only %d instead of %d bytes...\n\n", cb_read, cb_take);
      break;
    }

    sectors_read += cb_read / SECTOR_SIZE;
    if (fwrite(buf, 1, cb_read, fpo) != cb_read) {
      perror("Failed to write to output file");
      rc = -1;
      break;
    }

    /* progress */
    printf("\r%d sectors read", sectors_read);

  }

  DosClose(hf_disk);
  free(buf);

  return rc;
}

/******************************************************************************
 * usage() - print usage summary to STDOUT
 */
void usage(void)
{
  printf("readsect.exe - read HD sectors and store them to an output file.\n"
         "Call with a drive letter (for logical drive) or 1-based disk number (for\n"
         "physical drive)\n\n"
	 "Usage:\n\n"
	 "readsect <drive letter|number> <number of sectors> <outfile> [buffer size]\n\n"
         "where:\n"
         "drive letter         drive letter of logical drive to read from\n"
         "drive number         1-based physical disk number to read from\n"
         "number of sectors    number of sectors to read (e.g. 1024)\n"
         "outfile              path and filename of output file\n"
         "buffer size          buffer size in number of sectors (512 byte)\n"
         "                      (default is %d sectors)\n",
         SECTORS_PER_READ);
}


